package joyfill.table;

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.RemoveCircleOutline
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import joyfill.utils.Option2

private val shape = RoundedCornerShape(8.dp)

@OptIn(ExperimentalLayoutApi::class)
@Composable
internal fun FilterRow(
    state: ColumnState,
    onTextChange: (String?) -> Unit,
    onOptionChange: (Option2?) -> Unit,
    onSort: () -> Unit,
    onClearFilter: () -> Unit,
    onCapture: (suspend () -> String?)? = null,
) = AnimatedVisibility(visible = state.current != null) {
    FlowRow(
        horizontalArrangement = Arrangement.End,
        modifier = Modifier
            .fillMaxWidth()
            .padding(bottom = 4.dp)
            .background(shape = shape, color = LocalContentColor.current.copy(alpha = 0.2f))
            .padding(4.dp)
    ) {
        when (val filter = state.current) {
            is ColumnFilter.TextColumnFilter -> SearchTextInput(
                value = filter.value,
                onChange = onTextChange,
                modifier = Modifier
                    .weight(1f)
                    .background(shape = shape, color = MaterialTheme.colorScheme.surface)
            )

            is ColumnFilter.NumberColumnFilter -> SearchNumberInput(
                value = filter.value?.toString(),
                onChange = onTextChange,
                modifier = Modifier
                    .weight(1f)
                    .background(shape = shape, color = MaterialTheme.colorScheme.surface)
            )

            is ColumnFilter.DropdownColumnFilter -> SearchDropDown(
                options = filter.column.options,
                value = buildList {
                    val option = filter.value
                    if (option != null) add(option)
                },
                onAdd = onOptionChange,
                onRemove = { onOptionChange(null) },
                readonly = false,
                multiple = false,
                modifier = Modifier.weight(1f)
                    .fillMaxRowHeight()
                    .background(shape = shape, color = MaterialTheme.colorScheme.surface),
            )

            is ColumnFilter.MultiselectColumnFilter -> SearchDropDown(
                options = filter.column.options,
                value = buildList {
                    val option = filter.value
                    if (option != null) add(option)
                },
                onAdd = onOptionChange,
                onRemove = { onOptionChange(null) },
                readonly = false,
                multiple = false,
                modifier = Modifier.weight(1f)
                    .fillMaxRowHeight()
                    .background(shape = shape, color = MaterialTheme.colorScheme.surface),
            )

            is ColumnFilter.BarcodeColumnFilter -> SearchBarcodeInput(
                value = filter.value,
                onChange = onTextChange,
                onCapture = onCapture,
                modifier = Modifier
                    .weight(1f)
                    .background(shape = shape, color = MaterialTheme.colorScheme.surface)
            )

            else -> {}
        }

        Spacer(modifier = Modifier.width(4.dp))

        SortView(
            option = state.current?.sort ?: SortingOption.None,
            modifier = Modifier
                .clickable(onClick = onSort)
                .background(shape = shape, color = MaterialTheme.colorScheme.surface)
                .padding(horizontal = 4.dp, vertical = 5.dp)
        )

        Spacer(modifier = Modifier.width(4.dp))

        Icon(
            imageVector = Icons.Filled.RemoveCircleOutline,
            contentDescription = "Cancel",
            modifier = Modifier
                .fillMaxRowHeight()
                .clickable(onClick = onClearFilter)
                .background(shape = shape, color = MaterialTheme.colorScheme.surface)
                .padding(horizontal = 4.dp)
        )
    }
}