package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusState
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.input.KeyboardCapitalization
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.editors.TextFieldEditor
import joyfill.shared.RawInputField
import joyfill.utils.rememberAutoCleaner

@Composable
internal fun JoyTextField(
    editor: TextFieldEditor,
    mode: Mode,
    onSignal: (Signal<String?>) -> Unit,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    Column(modifier = Modifier.testTag(editor.field.id).fillMaxWidth()) {
        val field = remember(editor) { editor.field }
        JoyFieldHead(field)
        Spacer(modifier = Modifier.height(2.dp))
        var value by remember(editor) { mutableStateOf(field.value) }
        val focus = remember(onSignal, editor) { FocusManager(onSignal) { editor.value = value } }

        var clean by editor.rememberAutoCleaner(value)

        RawTextField(
            value = value,
            borders = true,
            maxLines = 1,
            readonly = field.disabled || mode == Mode.readonly,
            modifier = Modifier.testTag("${field.id}-body").fillMaxWidth(),
            onChange = {
                field.value = it
                clean = false
                value = it
                onSignal(Signal.Change(value))
            },
            onFocusChanged = focus.handler
        )
        Spacer(modifier = Modifier.height(8.dp))
    }
}

@Composable
internal fun RawTextField(
    value: String?,
    placeholder: String? = null,
    borders: Boolean = true,
    modifier: Modifier = Modifier,
    onChange: (String) -> Unit,
    maxLines: Int = Int.MAX_VALUE,
    readonly: Boolean = false,
    minLines: Int = 1,
    onFocusChanged: (FocusState) -> Unit,
) = RawInputField(
    value = value,
    placeholder = placeholder,
    borders = borders,
    modifier = modifier,
    onChange = onChange,
    maxLines = maxLines,
    readonly = readonly,
    keyboard = KeyboardOptions.Default.copy(capitalization = KeyboardCapitalization.Sentences),
    minLines = minLines,
    onFocusChanged = onFocusChanged
)

