package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.OutlinedTextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusState
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.input.KeyboardType
import cinematic.watchAsState
import com.ionspin.kotlin.bignum.decimal.BigDecimal
import joyfill.editors.NumberFieldEditor
import joyfill.fields.Field
import joyfill.shared.RawInputField
import joyfill.utils.rememberAutoCleaner

@Composable
internal fun JoyNumberField(
    editor: NumberFieldEditor,
    mode: Mode,
    onSignal: (Signal<Double?>) -> Unit,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    val field = remember(editor) { editor.field }
    var value by remember {
        val num = try { field.value?.toString() ?: "" } catch (_: Exception) { "" }
        mutableStateOf(num)
    }

    var clean by editor.rememberAutoCleaner(value.toDoubleOrNull())
    val focus = remember(onSignal) { FocusManager(onSignal) { editor.value = value.toDoubleOrNull() } }
    Column(modifier = Modifier.fillMaxWidth()) {
        JoyFieldHead(field)
        OutlinedTextField(
            value = value,
            onValueChange = {
                if (!it.isValidNumber()) return@OutlinedTextField
                val v = it
                value = it
                field.value = v.toDoubleOrNull()
                clean = false
                onSignal(Signal.Change(v.toDoubleOrNull()))
            },
            readOnly = field.disabled || mode == Mode.readonly,
            modifier = Modifier.testTag("${field.id}-body").fillMaxWidth().onFocusChanged(focus.handler),
            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal)
        )
    }
}



@Composable
internal fun RawNumberField(
    value: Double?,
    placeholder: String? = null,
    borders: Boolean = true,
    modifier: Modifier = Modifier,
    onChange: (Double?) -> Unit = {},
    maxLines: Int = Int.MAX_VALUE,
    readonly: Boolean = false,
    minLines: Int = 3,
    onFocusChanged: (FocusState) -> Unit = {},
)  = RawInputField(
    columnType = Field.Type.number,
    value = BigDecimal.fromDouble(value ?: 0.0).toPlainString(),
    placeholder = placeholder,
    borders = borders,
    modifier = modifier,
    onChange = { it.toDoubleOrNull()?.let { v -> onChange(v) } },
    maxLines = maxLines,
    readonly = readonly,
    keyboard = KeyboardOptions.Default.copy(keyboardType = KeyboardType.Decimal),
    minLines = minLines,
    onFocusChanged = onFocusChanged
)

fun String.isValidNumber() = all {
    it.isDigit() || it == '.' || it == '-' || it == ','
}