package joyfill

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.focusable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import cinematic.watchAsState
import joyfill.editors.BarcodeFieldEditor
import joyfill.legacy_compose.generated.resources.Res
import joyfill.legacy_compose.generated.resources.barcode_scanner
import kotlinx.coroutines.launch
import org.jetbrains.compose.resources.painterResource

@Composable
internal fun JoyBarcodeField(
    page: Page,
    editor: BarcodeFieldEditor,
    mode: Mode,
    onCapture: (suspend (FieldEvent) -> String?)? = null,
) = AnimatedVisibility(visible = !editor.hidden.watchAsState()) {
    Column(modifier = Modifier.testTag(editor.field.id).fillMaxWidth()) {
        val field = remember(editor) { editor.field }
        JoyFieldHead(field)
        Spacer(modifier = Modifier.height(2.dp))
        var value by remember(editor) { mutableStateOf(field.value) }

        val barcodeHandler = if (onCapture != null) {
            suspend {
                val event = FieldEvent(
                    field = field,
                    page = page,
                )
                onCapture(event)
            }
        } else null

        RawBarcodeField(
            value = value,
            borders = true,
            maxLines = 1,
            readonly = field.disabled || mode == Mode.readonly,
            modifier = Modifier.testTag("${field.id}-body").fillMaxWidth(),
            onChange = {
                field.value = it
                value = it
            },
            onCapture = barcodeHandler,
        )
        Spacer(modifier = Modifier.height(8.dp))
    }
}

@Composable
internal fun RawBarcodeField(
    value: String?,
    placeholder: String? = null,
    borders: Boolean = true,
    onChange: (String?) -> Unit,
    modifier: Modifier = Modifier,
    onCapture: (suspend () -> String?)?,
    maxLines: Int = Int.MAX_VALUE,
    readonly: Boolean = false,
    keyboard: KeyboardOptions = KeyboardOptions.Default.copy(keyboardType = KeyboardType.Text),
    minLines: Int = 1,
) {
    var v by remember(value) { mutableStateOf(value ?: "") }
    val scope = rememberCoroutineScope()

    Box(modifier = modifier) {
        OutlinedTextField(
            value = v,
            onValueChange = {
                v = it
                onChange(it)
            },
            readOnly = readonly,
            modifier = Modifier.fillMaxWidth().padding(4.dp),
            singleLine = maxLines == 1,
            maxLines = maxLines,
            placeholder = {
                if (placeholder != null) Text(
                    placeholder,
                    color = LocalContentColor.current.copy(alpha = 0.6f)
                )
            },
            keyboardOptions = keyboard,
            minLines = minLines,
            colors = if (borders) {
                OutlinedTextFieldDefaults.colors()
            } else {
                OutlinedTextFieldDefaults.colors(
                    focusedBorderColor = Color.Companion.Transparent,
                    unfocusedBorderColor = Color.Companion.Transparent,
                )
            },
            trailingIcon = {
                Icon(
                    painter = painterResource(Res.drawable.barcode_scanner),
                    contentDescription = "Scan",
                    modifier = Modifier.focusable(false).clickable {
                        scope.launch {
                            onCapture?.invoke()?.let {
                                v = it
                                onChange(it)
                            }
                        }
                    }
                )
            }
        )
    }
}

