package joyfill.internal

import joyfill.BarcodeField
import joyfill.ChartField
import joyfill.DateField
import joyfill.DocumentBuilder
import joyfill.DropdownField
import joyfill.FieldPosition
import joyfill.File
import joyfill.FileField
import joyfill.IdentityGenerator
import joyfill.ImageField
import joyfill.LogicBuilder
import joyfill.MultiSelectField
import joyfill.MutableDocument
import joyfill.MutablePage
import joyfill.NumberField
import joyfill.Page
import joyfill.SignatureField
import joyfill.TableField
import joyfill.TextArea
import joyfill.TextField
import joyfill.TextLabel
import joyfill.ToolTip
import joyfill.UnknownField
import joyfill.chart.LineBuilder
import joyfill.chart.LineBuilderImpl
import joyfill.fields.Field
import joyfill.fields.VField
import joyfill.fields.chart.Axis
import joyfill.fields.chart.Line
import joyfill.table.ColumnBuilder
import joyfill.table.TableColumnsBuilderImpl
import joyfill.utils.Attachment
import joyfill.utils.Option

class DocumentBuilderImpl(
    internal val document: MutableDocument,
    private val identity: IdentityGenerator
) : DocumentBuilder {

    private val file by lazy {
        document.files.getOrNull(0) ?: File(
            id = identity.generate(),
            name = document.name,
            pages = mutableListOf(),
            pageOrder = mutableListOf()
        ).also { document.files.add(it) }
    }

    private var cursor: MutablePage? = null

    override fun name(value: String) {
        document.name = value
        file.name = value
    }

    private fun cursor(): MutablePage {
        return cursor ?: page("New Page")
    }

    override fun page(
        name: String?,
        hidden: Boolean,
        logic: (LogicBuilder.() -> Unit)?
    ): MutablePage {
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        val p = Page(
            id = identity.generate(),
            name = name ?: "Page ${file.pages.size + 1}",
            hidden = hidden,
            positions = mutableListOf(),
            logic = lgc
        )
        cursor = p
        file.pages.add(p)
        file.pageOrder.add(p.id)
        return p
    }

    private fun add(field: VField) {
        val position = FieldPosition(
            id = identity.generate(),
            field = field.id,
            displayType = "original",
            format = null,
            columns = mutableListOf()
        )
        add(field, position)
    }

    private fun add(field: VField, position: FieldPosition) {
        cursor().positions.add(position)
        document.fields.add(field)
    }

    private fun <F : VField> buildField(id: String?, builder: (uid: String) -> F) {
        add(builder(id ?: identity.generate()))
    }

    override fun text(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        TextField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, value, lgc)
    }

    override fun block(
        title: String,
        id: String?,
        identifier: String?,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?,
        logic: (LogicBuilder.() -> Unit)?
    )  = buildField(id) { uid ->
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        TextLabel(uid, title, identifier ?: "field-$uid", hidden, required, tip, value, lgc)
    }

    override fun textarea(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        TextArea(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, value, lgc)
    }

    override fun signature(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        SignatureField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, value, lgc)
    }

    override fun number(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: Number?,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        NumberField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, value, lgc)
    }

    override fun date(
        title: String,
        id: String?,
        identifier: String?,
        format: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: Long?,
        logic: (LogicBuilder.() -> Unit)?
    ) {
        val uid = id ?: identity.generate()
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        val field = DateField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, format, tip, value, lgc)
        val position = FieldPosition(
            id = identity.generate(),
            field = field.id,
            displayType = "original",
            format = format,
            columns = mutableListOf()
        )
        add(field, position)
    }

    private fun String.toOption() = Option(id = identity.generate(), value = this)

    override fun dropdown(
        title: String,
        options: List<String>,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val o = options.map { it.toOption() }
        val selected = o.firstOrNull { it.value == value || it.id == value }
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        DropdownField(uid, title, identifier ?: "field-$uid", o, readonly, hidden, required, selected, lgc)
    }

    override fun select(
        title: String,
        options: List<String>,
        multi: Boolean,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: List<String>,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val selected = mutableListOf<String>()
        val o = options.map {
            val out = it.toOption()
            if (value.contains(it)) {
                selected.add(out.id)
            }
            out
        }
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        MultiSelectField(uid, title, multi, identifier ?: "field-$uid", o, hidden, required, tip, selected, lgc)
    }

    override fun select(
        title: String,
        options: List<String>,
        multi: Boolean,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String,
        logic: (LogicBuilder.() -> Unit)?
    ) = select(title, options, multi, id, identifier, readonly, hidden, required, tip, listOf(value))

    private fun String.toAttachment() = Attachment(id = identity.generate(), url = this)

    override fun image(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: List<String>,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val attachments = value.map { it.toAttachment() }
        ImageField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, attachments)
    }

    override fun barcode(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid ->
        val lgc = LogicBuilderImpl().apply { logic?.invoke(this) }.logic
        BarcodeField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, value, lgc)
    }

    override fun image(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String,
        logic: (LogicBuilder.() -> Unit)?
    ) = image(title, id, identifier, readonly, hidden, required, tip, listOf(value))

    override fun file(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: List<String>,
        logic: (LogicBuilder.() -> Unit)?
    ) = buildField(id) { uid -> FileField(uid, title, identifier ?: "field-$uid", readonly, hidden, required, tip, value.map { it.toAttachment() }) }

    override fun table(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        tip: ToolTip,
        logic: (LogicBuilder.() -> Unit)?,
        columns: (ColumnBuilder.() -> Unit)?
    ){
        val uid = id ?: identity.generate()
        val builder = TableColumnsBuilderImpl(identity = identity)
        columns?.invoke(builder)
        val table = TableField(
            id = uid,
            title = title,
            identifier = identifier ?: "field-$uid",
            columns = builder.columns,
        )

        val position = FieldPosition(
            id = identity.generate(),
            field = uid,
            displayType = "original",
            format = null,
            columns = builder.positions
        )

        add(table, position)
    }

    override fun chart(
        title: String,
        y: Axis,
        x: Axis,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        tip: ToolTip,
        logic: (LogicBuilder.() -> Unit)?,
        lines: (LineBuilder.() -> Unit)?
    ) = chart(
        title, y, x, id, identifier, readonly, hidden,
        lines = LineBuilderImpl(identity).also { lines?.invoke(it) }.lines
    )

    override fun chart(
        title: String,
        y: Axis,
        x: Axis,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        lines: List<Line>,
        tip: ToolTip,
        logic: (LogicBuilder.() -> Unit)?,
    ) = buildField(id) { uid ->
        ChartField(
            id = uid,
            title = title,
            identifier = identifier ?: "field-$uid",
            y = y,
            x = x,
            lines = lines
        )
    }

    override fun unknown(
        title: String,
        id: String?,
        identifier: String?,
        hidden: Boolean,
        tip: ToolTip
    ) = buildField(id) { uid ->
        UnknownField(uid, title, identifier ?: "field-$uid", Field.Type.unknown)
    }
}