package joyfill.table


import joyfill.IdentityGenerator
import joyfill.LogicBuilder
import joyfill.ToolTip
import joyfill.fields.Field
import joyfill.fields.table.BarcodeColumn
import joyfill.fields.table.BlockColumn
import joyfill.fields.table.Column
import joyfill.fields.table.ColumnPosition
import joyfill.fields.table.DateColumn
import joyfill.fields.table.DropdownColumn
import joyfill.fields.table.ImageColumn
import joyfill.fields.table.MultiselectColumn
import joyfill.fields.table.NumberColumn
import joyfill.fields.table.TextColumn
import joyfill.toColumn
import joyfill.toColumnPosition
import joyfill.utils.Attachment
import joyfill.utils.ID
import joyfill.utils.Option
import joyfill.utils.Option2

internal class TableColumnsBuilderImpl(
    private val identity: IdentityGenerator
) : ColumnBuilder {
    val columns = mutableListOf<Column>()
    val positions = mutableListOf<ColumnPosition>()

    override fun text(title: String, id: String?, required: Boolean?, value: String?, hidden: Boolean) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.text.name,
            Column::required.name to required,
            TextColumn::value.name to value
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun block(title: String, id: String?, value: String?, hidden: Boolean) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.block.name,
            BlockColumn::value.name to value
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun date(
        title: String,
        id: String?,
        identifier: String?,
        format: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: Long?,
        logic: (LogicBuilder.() -> Unit)?
    ) {

        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.date.name,
            Column::required.name to required,
            DateColumn::value.name to value,
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::format.name to format,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun number(title: String, id: String?, required: Boolean?,value: Double?, hidden: Boolean) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.number.name,
            Column::required.name to required,
            NumberColumn::value.name to value
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun dropdown(title: String, options: List<String>, id: String?, required: Boolean?, value: String?, hidden: Boolean) {
        var default: Option2? = null
        val opts = options.map {
            val option = Option(identity.generate(), it)
            if (it == value) {
                default = option
            }
            option
        }

        val column = mutableMapOf(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.dropdown.name,
            Column::required.name to required,
            DropdownColumn::options.name to opts.map { it.toMap() }.toMutableList(),
            DropdownColumn::value.name to default?.id
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun multiselect(
        title: String,
        options: Map<String, String?>,
        multi: Boolean,
        id: String?,
        required: Boolean?,
        value: List<String>,
        hidden: Boolean
    ) {
        var default = mutableListOf<Option2>()
        val opts = options.map {
            val option = Option(identity.generate(), it.key, null, it.value)
            if (it.key in value) {
                default.add(option)
            }
            option
        }

        val column = mutableMapOf(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.multiSelect.name,
            Column::required.name to required,
            MultiselectColumn::multi.name to multi,
            MultiselectColumn::options.name to opts.map { it.toMap() }.toMutableList(),
            MultiselectColumn::value.name to default.map { it.id }
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun image(title: String, id: String?, identifier: String?, required: Boolean?, value: List<String>, hidden: Boolean) {
        val attachments = value.map { Attachment(id = identity.generate(), url = it) }
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.image.name,
            Column::required.name to required,
            ImageColumn::value.name to attachments.map { it.toMap() }.toMutableList()
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun barcode(title: String, id: String?, required: Boolean?, value: String?, hidden: Boolean) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.barcode.name,
            Column::required.name to required,
            BarcodeColumn::value.name to value
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun signature(
        title: String,
        id: String?,
        identifier: String?,
        readonly: Boolean,
        hidden: Boolean,
        required: Boolean,
        tip: ToolTip,
        value: String?
    ) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.signature.name,
            Column::required.name to required,
            ImageColumn::value.name to value
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun progress(title: String, id: String?, value: String?, hidden: Boolean) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.progress.name,
        ).toColumn()

        val position = mutableMapOf<String, Any?>(
            ID to column.id,
            ColumnPosition::hidden.name to hidden
        ).toColumnPosition()

        positions.add(position)
        columns.add(column)
    }

    override fun unknown(
        title: String,
        id: String?,
        required: Boolean?,
        identifier: String?,
    ) {
        val column = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Column::title.name to title,
            Column::type.name to Field.Type.unknown.name,
            Column::required.name to required,
        ).toColumn()
        columns.add(column)
    }
}