package joyfill.internal

import io.ktor.client.request.HttpRequestBuilder
import io.ktor.client.request.accept
import io.ktor.client.request.delete
import io.ktor.client.request.get
import io.ktor.client.request.headers
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.client.statement.bodyAsText
import io.ktor.http.ContentType
import io.ktor.http.contentType
import joyfill.internal.config.ApiConfig
import kotlinx.serialization.builtins.MapSerializer
import kotlinx.serialization.builtins.serializer

internal abstract class AbstractJoyfillApi(protected val config: ApiConfig) {
    protected fun Map<String, Any?>.toUrl(path: String) = buildString {
        append("${config.url}/${config.version}/$path?")
        val params = filterValues {
            it?.toString()?.isEmpty() == false
        }.entries.joinToString("&") { (k, v) -> "$k=$v" }
        append(params)
    }

    private fun HttpRequestBuilder.addJoyHeaders() = headers {
        append("Authorization", config.authorizationHeader)
        accept(ContentType.Application.Json)
    }

    protected suspend fun Map<String, Any?>.joyGet(path: String): String = config.http.get(toUrl(path)) { addJoyHeaders() }.bodyAsText()

    protected suspend fun Map<String, String>.joyPost(path: String): String {
        val json = config.codec.encodeToString(MapSerializer(String.serializer(), String.serializer()), this)
        return json.joyPost(path)
    }

    protected suspend fun String.joyPost(path: String): String = config.http.post("${config.url}/${config.version}/$path") {
        addJoyHeaders()
        contentType(ContentType.Application.Json)
        setBody(this@joyPost)
    }.bodyAsText()

    protected suspend fun joyDel(path: String) = config.http.delete("${config.url}/${config.version}/$path") { addJoyHeaders() }.bodyAsText()
}