package joyfill.internal

import joyfill.Document
import joyfill.DocumentsApi
import joyfill.JoyStage
import joyfill.internal.config.ApiConfig
import joyfill.toDocument
import koncurrent.Later
import koncurrent.coroutines.asCoroutineScope
import koncurrent.later
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonObject

internal class DocumentsApiImpl(config: ApiConfig) : AbstractJoyfillApi(config), DocumentsApi {

    private val scope by lazy { config.executor.asCoroutineScope() }

    override fun create(document: Document): Later<Document> = scope.later {
        val json = mapOf(
            "stage" to JoyStage.published.name,
            "template" to document.identifier,
        ).joyPost("documents")
        json.toDocument()
    }

    override fun update(document: Document): Later<Document> = scope.later {
        val outgoing = document.toJsonString()
        val incoming = outgoing.joyPost("documents/${document.identifier}")
        incoming.toDocument()
    }

    override fun list(
        limit: Int,
        page: Int,
        template: String?,
        group: String?,
        stage: JoyStage,
    ): Later<List<Document>> = scope.later {
        if (limit !in 1..100) throw IllegalArgumentException("limit must be between 1 and 100")
        val json = mapOf(
            "limit" to limit,
            "page" to page,
            "template" to template,
            "stage" to stage,
            "group" to group
        ).joyGet("documents")
        (config.codec.decodeFromString<JsonObject>(json)["data"] as JsonArray).map {
            (it as JsonObject).toDocument()
        }
    }

    override fun delete(identifier: String): Later<String> = scope.later {
        joyDel("documents/$identifier")
        identifier
    }
}