package joyfill.editors.table.utils

import com.ionspin.kotlin.bignum.decimal.BigDecimal
import joyfill.editors.barcode.BarcodeEditor
import joyfill.editors.components.ComponentEditor
import joyfill.editors.dropdown.DropdownEditor
import joyfill.editors.multi_select.MultiSelectEditor
import joyfill.editors.number.NumberEditor
import joyfill.editors.table.ColumnFilter
import joyfill.editors.table.RowManager
import joyfill.editors.text.TextEditor

internal fun <M : RowManager> List<M>.applyFilters(filters: List<ColumnFilter>): List<M> {
    return if (filters.isEmpty()) this
    else filter { it.satisfy(filters) }
}

internal fun <M : RowManager> M.satisfy(filters: List<ColumnFilter>): Boolean =
    editorsFor(filters)
        .all { (filter, editor) -> editor.satisfies(filter) }

internal fun <M : RowManager> M.editorsFor(filters: List<ColumnFilter>) = filters.mapNotNull { filter ->
    val editor = find(filter.column?.id)
    if (editor == null) null
    else filter to editor
}

internal fun ComponentEditor.satisfies(filter: ColumnFilter): Boolean {
    if (filter.value == null || filter.column == null) return true
    return when (filter) {
        is ColumnFilter.EmptyColumnFilter -> false
        is ColumnFilter.BarcodeColumnFilter -> with(this as BarcodeEditor) {
            val data = state.value.data ?: return false
            data.contains(filter.value!!, ignoreCase = true)
        }

        is ColumnFilter.DropdownColumnFilter -> with(this as DropdownEditor) {
            val data = state.value.data?.id ?: return false
            data == filter.value!!.id
        }

        is ColumnFilter.MultiselectColumnFilter -> with(this as MultiSelectEditor) {
            val data = state.value.data.ifEmpty { return false }
            data.any { it.id == filter.value!!.id }
        }

        is ColumnFilter.NumberColumnFilter -> with(this as NumberEditor) {
            val data = state.value.data ?: return false
            BigDecimal.fromDouble(data).toPlainString().startsWith(filter.value!!)
        }

        is ColumnFilter.TextColumnFilter -> with(this as TextEditor) {
            val data = state.value.data ?: return false
            data.contains(filter.value!!, ignoreCase = true)
        }
    }
}
