package joyfill.editors.table.internal

import joyfill.barcode.BarcodeColumn
import joyfill.block.BlockColumn
import joyfill.date.DateColumn
import joyfill.dropdown.DropdownColumn
import joyfill.editors.table.RowManager
import joyfill.events.ChangeLog
import joyfill.events.FieldUpdateChange
import joyfill.events.RowCreateChange
import joyfill.events.RowDeleteChange
import joyfill.events.RowUpdateChange
import joyfill.file.FileColumn
import joyfill.image.ImageColumn
import joyfill.multi_select.MultiSelectColumn
import joyfill.number.NumberColumn
import joyfill.signature.SignatureColumn
import joyfill.table.Column
import joyfill.table.Row
import joyfill.table.TableSchema
import joyfill.table.toCells
import joyfill.text.TextColumn
import joyfill.text_area.TextAreaColumn
import joyfill.utils.TIME_ZONE

internal class TableChangeLogBuilder(
    private val columns: List<Column>
) {
    fun buildRowCreateChangeValue(
        rowManager: RowManager,
        index: Int,
        includeChildren: Boolean
    ): MutableMap<String, Any?> {
        val newCells = columns.toCells(
            columns.flatMap { column ->
                listOf(
                    column.id to when (column) {
                        is TextColumn -> column.value
                        is TextAreaColumn -> column.value
                        is BarcodeColumn -> column.value
                        is NumberColumn -> column.value
                        is DateColumn -> column.value
                        is DropdownColumn -> column.value
                        is SignatureColumn -> column.value
                        is BlockColumn -> column.value
                        is ImageColumn -> column.value.map { it.toMap() }.toMutableList()
                        is FileColumn -> column.value.map { it.toMap() }.toMutableList()
                        is MultiSelectColumn -> column.value.toMutableList()
                        else -> null
                    }
                ).filter {
                    when (val value = it.second) {
                        null -> false
                        is MutableList<*> -> value.isNotEmpty()
                        else -> true
                    }
                }
            }.toMap().toMutableMap()
        )

        val newRowMap = rowManager.row.toMap().toMutableMap().apply {
            if (!includeChildren)
                remove(Row::children.name)
            if (!columns.any { it is DateColumn })
                remove(TIME_ZONE)
            remove(Row::deleted.name)
            this[Row::cells.name] = newCells.toMap()
        }
        return mutableMapOf(
            RowCreateChange::targetRowIndex.name to index,
            RowCreateChange::row.name to newRowMap,
        )
    }

    fun buildRowMoveChangeValue(row: Row?, index: Int): MutableMap<String, Any?> =
        mutableMapOf(
            RowDeleteChange::rowId.name to row?.id,
            RowCreateChange::targetRowIndex.name to index
        )

    fun buildRowDeleteChangeValue(rowManager: RowManager): MutableMap<String, Any?> =
        mutableMapOf(
            RowDeleteChange::rowId.name to rowManager.row.id
        )

    fun buildRowUpdateChangeValue(columnId: String, row: Row, changeLog: ChangeLog): MutableMap<String, Any?> {
        val value =
            (changeLog.toMap()[ChangeLog::change.name] as? Map<*, *>)?.get(FieldUpdateChange::value.name)
        val id = changeLog.fieldId
        val singleCellMap = mutableMapOf(id to value)
        val newRowMap = row.toMap().toMutableMap().apply {
            this[Row::cells.name] = singleCellMap
            if (row.cells.date(columnId) == null)
                remove(TIME_ZONE)
            remove(Row::deleted.name)
            remove(Row::children.name)
        }
        return mutableMapOf(
            RowUpdateChange::columnId.name to columnId,
            RowUpdateChange::rowId.name to row.id,
            RowUpdateChange::row.name to newRowMap,
            TableSchema::columns.name to columns
        )
    }
} 
