package joyfill.editors.multi_select.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.collections.PageCollection
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.multi_select.MultiSelectEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.notify
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.multi_select.MultiSelectComponent
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import joyfill.utils.Option
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class MultiSelectEditorImpl(
    override val component: MultiSelectComponent,
    override val options: List<Option>,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String = component.id,
    dispatcher: EventDispatcher?,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
) : MultiSelectEditor,
    AbstractComponentEditor(
        document,
        pages,
        component,
        onValidate,
        onChange,
        identifier,
        fieldId,
        dispatcher,
        dependents,
        resolver,
        library
    ) {

    private fun look(key: String?): Option? =
        options.firstOrNull { it.id == key || it.value == key }

    override val state by lazy {
        val field = component as? joyfill.Field

        val initialValue = if (field != null && field.formulas.isNotEmpty()) {
            library?.call(field.id, resolver) as? MutableList<String>
        } else {
            component.value
        }
        if (initialValue == null) {
            value(null)
        } else {
            val value = options.filter { it.id in initialValue || it.value in initialValue }
            value(value)
        }
        mutableLiveOf(
            MultiSelectEditorStateImpl(
                data = options.filter { it.id in component.value }.toMutableList(),
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate(),
            )
        )
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override fun value(v: List<Option>?, origin: Origin) {
        // Check if the new options are the same as current component value
        val newOptionIds = v?.map { it.id }?.toSet() ?: emptySet()
        val currentIds = component.value.toSet()
        if (newOptionIds == currentIds) return

        // Set the component value first
        component.value.clear()
        val newOptions = v?.mapNotNull { incomingOption ->
            options.find { it.id == incomingOption.id || it.value == incomingOption.value }
        } ?: emptyList()

        component.value.addAll(newOptions.map { it.id })

        // Update state and notify
        updateStateAndNotify(newOptions.toMutableList(), origin)
        updateDependentsValues()
    }

    override fun canSelectMultiple(): Boolean = component.multi

    override fun select(key: String?, origin: Origin) {
        val option = look(key) ?: return
        val currentSelection = selected()

        if (currentSelection.any { it.id == option.id }) return

        val newSelection = if (component.multi) {
            currentSelection + option
        } else {
            listOf(option)
        }

        value(newSelection, origin)
    }

    override val isValid: Boolean
        get() = component.value.isNotEmpty()

    private fun updateStateAndNotify(data: MutableList<Option>, origin: Origin = Origin.User) {
        val currentIds = state.value.data.map { it.id }.toSet()
        val newIds = data.map { it.id }.toSet()
        if (currentIds == newIds) return

        resolveConditions()
        state.value = state.value.copy(
            data = data,
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        if (origin.notify) notifyChange(component.value)
    }

    override fun select(item: Option?, origin: Origin): Option? {
        select(item?.id, origin)
        return item
    }

    override fun unselect(key: String?, origin: Origin) {
        val option = look(key) ?: return
        val currentSelection = selected()

        if (currentSelection.none { it.id == option.id }) return

        val newSelection = currentSelection.filter { it.id != option.id }
        value(newSelection, origin)
    }

    override fun unselect(item: Option?, origin: Origin): Option? {
        unselect(item?.id, origin)
        return item
    }

    private fun selected(): List<Option> {
        val found = mutableListOf<Option>()
        val value = this.component.value
        for (option in value) found.add(look(option) ?: continue)
        return found
    }

    override fun set(options: List<Option>) {
        value(options, Origin.User)
    }
}