package joyfill.editors.file.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collections.PageCollection
import joyfill.components.internal.AbstractFileComponent
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.file.AbstractFileEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.notify
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import joyfill.utils.Attachment
import wisdom.ResolutionResource
import wisdom.ast.Library

internal abstract class AbstractFileEditorImpl<T : AbstractFileComponent>(
    override val component: T,
    private val identity: IdentityGenerator,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
    dispatcher: EventDispatcher?,
) : AbstractComponentEditor(
    document,
    pages,
    component,
    onValidate,
    onChange,
    identifier,
    fieldId,
    dispatcher,
    dependents,
    resolver,
    library,
),
    AbstractFileEditor {

    override val state by lazy {
        mutableLiveOf(
            FileEditorStateImpl(
                data = component.value,
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate()
            )
        )
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override fun value(v: List<Attachment>?, origin: Origin) {
        if (component.value == v) return

        // Set the component value first
        component.value.clear()
        v?.let { component.value.addAll(it) }

        // Update conditions and state
        updateStateAndNotify(component.value, origin)
        updateDependentsValues()
    }

    private fun updateStateAndNotify(data: List<Attachment>, origin: Origin = Origin.User) {
        resolveConditions()
        state.value = state.value.copy(
            data = data.map { it.copy() },
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        if (origin.notify) {
            notifyChange(
                data.map {
                    it.copy().toMap().apply {
                        remove(Attachment::fileName.name)
                        remove(Attachment::filePath.name)
                        remove(Attachment::download.name)
                        put(Attachment::url.name, it.url.take(2000))
                    }
                }
            )
        }
    }

    override fun add(url: String) = add(listOf(url))
    override fun add(attachment: Attachment) = add(listOf(attachment), Origin.User)


    override fun add(attachments: List<Attachment>, origin: Origin) {
        if (attachments.isEmpty()) return
        if (component.multi) {
            component.value.addAll(attachments)
        } else {
            component.value.clear()
            component.value.add(attachments.first())
        }
        updateStateAndNotify(component.value, origin)
    }

    override fun add(urls: Collection<String>) {
        val attachments = urls.map { url ->
            Attachment(
                id = identity.generate(),
                url = url,
            )
        }

        add(attachments, Origin.User)
    }

    override fun set(attachments: List<Attachment>) {
        component.value.clear()
        component.value.addAll(attachments)
        updateStateAndNotify(component.value)
    }

    override fun set(urls: Collection<String>) {
        val attachments = urls.map { url ->
            Attachment(
                id = identity.generate(),
                url = url,
            )
        }

        set(attachments)
    }

    override fun remove(attachment: Attachment?, origin: Origin) {
        attachment?.let {
            remove(listOf(it), origin)
        }
    }

    override fun remove(key: String?) {
        key?.let {
            remove(listOf(it))
        }
    }

    override val isValid: Boolean
        get() = component.value.isNotEmpty()

    override fun remove(attachments: List<Attachment>, origin: Origin) {
        val ids = attachments.map { it.id }
        val urlsList = attachments.map { it.url }
        val fileNames = attachments.mapNotNull { it.fileName }
        val found = component.value.filter {
            it.id in ids || it.fileName in fileNames || it.url in urlsList
        }
        if (!found.isNotEmpty()) return
        component.value.removeAll(found)
        updateStateAndNotify(component.value, origin)
    }

    override fun remove(keys: Collection<String>) {
        val found = component.value.filter {
            it.id in keys || it.fileName in keys || it.url in keys
        }
        if (found.isEmpty()) return
        component.value.removeAll(found)
        updateStateAndNotify(component.value)
    }

    override fun canSelectMultiple(): Boolean = component.multi
}