package joyfill.editors.document.helper

import joyfill.editors.collection.CollectionEditor
import joyfill.editors.document.DocumentEditor
import joyfill.editors.table.Origin
import joyfill.editors.table.TableEditor
import joyfill.table.Column

/**
 * Updates a table row with new cell values.
 *
 * This function allows you to update specific cells in a table row. If no `rowId` is provided,
 * a new row will be created automatically. The function supports both regular table fields
 * and collection table fields.
 *
 * **Usage Examples:**
 * ```kotlin
 * // Update an existing row
 * editor.update(
 *     fieldId = "employees_table",
 *     rowId = "row_123",
 *     cellUpdates = mapOf(
 *         nameColumn to "John Doe",
 *         ageColumn to 30.0
 *     )
 * )
 *
 * // Create a new row (no rowId provided)
 * editor.update(
 *     fieldId = "employees_table",
 *     cellUpdates = mapOf(
 *         nameColumn to "Jane Smith",
 *         ageColumn to 25.0
 *     )
 * )
 * ```
 *
 * @param fieldId The unique identifier of the table field to update. This can be either
 *                a regular table field or a collection table field.
 * @param cellUpdates A map where keys are [Column] objects and values are the new cell values.
 *                    The values can be of any type supported by the column (String, Number, etc.).
 * @param rowId Optional identifier of the specific row to update. If `null`, a new row will be created.
 *              If provided, the row must exist in the table.
 * @param schemaId Optional schema identifier for collection fields. This is required when
 *                 working with collection table fields to specify which schema to use.
 *                 For regular table fields, this parameter is ignored.
 *
 * @throws IllegalArgumentException if the fieldId doesn't correspond to a valid table field.
 * @throws IllegalStateException if a rowId is provided but the row doesn't exist in the table.
 */
fun DocumentEditor.update(
    fieldId: String,
    cellUpdates: Map<Column, Any?>,
    rowId: String? = null,
    schemaId: String? = null,
) {
    val field = fields.find(fieldId)
    val tableEditor = field?.tableEditor(rowId, schemaId) ?: return
    val row = if (rowId == null) {
        tableEditor.rows.append(origin = Origin.Sdk)
    } else {
        tableEditor.rows.find(rowId)
    }
    cellUpdates.forEach { (column, newValue) ->
        row?.find(column.id)?.update(newValue)
    }
}

/**
 * Appends a new row to a table at a specified target index.
 *
 * This function creates a new row and inserts it at the specified index position. If the
 * target index is beyond the current number of rows, the new row will be appended at the end.
 * The function supports both regular table fields and collection table fields.
 *
 * **Usage Examples:**
 * ```kotlin
 * // Insert a new row at index 0 (beginning of table)
 * editor.append(
 *     fieldId = "employees_table",
 *     targetIndex = 0,
 *     cellUpdates = mapOf(
 *         nameColumn to "John Doe",
 *         ageColumn to 30.0
 *     )
 * )
 *
 * // Insert a new row after the first row (index 1)
 * editor.append(
 *     fieldId = "employees_table",
 *     targetIndex = 1,
 *     cellUpdates = mapOf(
 *         nameColumn to "Jane Smith",
 *         ageColumn to 25.0
 *     ),
 *     rowId = "custom_row_id"
 * )
 * ```
 *
 * @param fieldId The unique identifier of the table field to append to. This can be either
 *                a regular table field or a collection table field.
 * @param targetIndex The position where the new row should be inserted. If the index is
 *                    greater than the current number of rows, the row will be appended at the end.
 * @param cellUpdates A map where keys are [Column] objects and values are the new cell values.
 *                    The values can be of any type supported by the column (String, Number, etc.).
 * @param rowId Optional custom identifier for the new row. If not provided, a unique ID will be generated.
 * @param schemaId Optional schema identifier for collection fields. This is required when
 *                 working with collection table fields to specify which schema to use.
 *                 For regular table fields, this parameter is ignored.
 * @param parentPath Optional parent path for nested tables in collection fields. This is used
 *                   to specify the hierarchical structure when working with nested tables.
 *
 * @throws IllegalArgumentException if the fieldId doesn't correspond to a valid table field.
 * @throws IndexOutOfBoundsException if the targetIndex is negative.
 */
fun DocumentEditor.append(
    fieldId: String,
    targetIndex: Int,
    cellUpdates: Map<Column, Any?>,
    rowId: String? = null,
    schemaId: String? = null,
    parentPath: String? = null,
) {
    val field = fields.find(fieldId)
    val tableEditor = field?.tableEditor(schemaId = schemaId, parentPath = parentPath) ?: return
    val existingRow = tableEditor.rows.all().getOrNull(targetIndex - 1)?.row
    val row = existingRow?.let {
        tableEditor.rows.appendAfter(
            id = it.id,
            defaultId = rowId,
            origin = Origin.Sdk
        )
    }
        ?: tableEditor.rows.append(defaultId = rowId, origin = Origin.Sdk)
    cellUpdates.forEach { (column, newValue) ->
        row.find(column.id)?.update(newValue)
    }
}

/**
 * Moves a table row to a new position within the table.
 *
 * This function reorders rows within a table by moving a specific row to a new index position.
 * The function calculates the number of positions to move and uses the appropriate move operation
 * (up or down) to achieve the desired position.
 *
 * **Usage Examples:**
 * ```kotlin
 * // Move a row from position 2 to position 0 (beginning of table)
 * editor.move(
 *     fieldId = "employees_table",
 *     rowId = "row_123",
 *     toIndex = 0
 * )
 *
 * // Move a row from position 0 to position 3 (end of table)
 * editor.move(
 *     fieldId = "employees_table",
 *     rowId = "row_456",
 *     toIndex = 3
 * )
 * ```
 *
 * @param fieldId The unique identifier of the table field containing the row to move.
 *                This can be either a regular table field or a collection table field.
 * @param rowId The unique identifier of the row to move. The row must exist in the table.
 * @param toIndex The target index position where the row should be moved. If the index is
 *                greater than the current number of rows, the row will be moved to the end.
 * @param schemaId Optional schema identifier for collection fields. This is required when
 *                 working with collection table fields to specify which schema to use.
 *                 For regular table fields, this parameter is ignored.
 *
 * @throws IllegalArgumentException if the fieldId doesn't correspond to a valid table field.
 * @throws IllegalStateException if the rowId doesn't exist in the table.
 * @throws IndexOutOfBoundsException if the toIndex is negative.
 */
fun DocumentEditor.move(
    fieldId: String,
    rowId: String,
    toIndex: Int,
    schemaId: String? = null,
) {
    val field = fields.find(fieldId)
    val tableEditor = field?.tableEditor(rowId, schemaId) ?: return
    val rows = tableEditor.rows
    val originalRowIndex = rows.all().indexOfFirst { it.row.id == rowId }
    if (originalRowIndex == -1) return
    if (originalRowIndex > toIndex) {
        rows.up(rowId, originalRowIndex - toIndex, origin = Origin.Sdk)
    } else {
        rows.down(rowId, toIndex - originalRowIndex, origin = Origin.Sdk)
    }
    field.refresh(tableEditor)
}

/**
 * Removes a specific row from a table.
 *
 * This function permanently deletes a row from the table. The operation cannot be undone
 * and will trigger appropriate change events. The function supports both regular table fields
 * and collection table fields.
 *
 * **Usage Examples:**
 * ```kotlin
 * // Remove a specific row by its ID
 * editor.remove(
 *     fieldId = "employees_table",
 *     rowId = "row_123"
 * )
 *
 * // Remove a row from a collection table field
 * editor.remove(
 *     fieldId = "department_employees",
 *     rowId = "row_456",
 *     schemaId = "sales_department"
 * )
 * ```
 *
 * @param fieldId The unique identifier of the table field containing the row to remove.
 *                This can be either a regular table field or a collection table field.
 * @param rowId The unique identifier of the row to remove. The row must exist in the table.
 * @param schemaId Optional schema identifier for collection fields. This is required when
 *                 working with collection table fields to specify which schema to use.
 *                 For regular table fields, this parameter is ignored.
 *
 * @throws IllegalArgumentException if the fieldId doesn't correspond to a valid table field.
 * @throws IllegalStateException if the rowId doesn't exist in the table.
 */
fun DocumentEditor.remove(
    fieldId: String,
    rowId: String,
    schemaId: String? = null,
) {
    val field = fields.find(fieldId)
    val tableEditor = field?.tableEditor(rowId, schemaId) ?: return
    tableEditor.rows.delete(listOf(rowId), origin = Origin.Sdk)
    field.refresh(tableEditor)
}

/**
 * Retrieves the columns for a specific table field.
 *
 * This function returns a list of all columns defined in a table field. It supports both
 * regular table fields and collection table fields. For collection fields, you can specify
 * a schema to get columns for a specific table within the collection.
 *
 * **Usage Examples:**
 * ```kotlin
 * // Get columns from a regular table field
 * val columns = editor.columnsFor("employees_table")
 * columns?.forEach { column ->
 *     println("Column: ${column.title}, Type: ${column.type}")
 * }
 *
 * // Get columns from a collection table field with specific schema
 * val columns = editor.columnsFor(
 *     fieldId = "department_employees",
 *     schemaId = "sales_department"
 * )
 * ```
 *
 * @param fieldId The unique identifier of the table field to get columns from.
 *                This can be either a regular table field or a collection table field.
 * @param schemaId Optional schema identifier for collection fields. This is required when
 *                 working with collection table fields to specify which schema to use.
 *                 For regular table fields, this parameter is ignored.
 *
 * @return A list of [Column] objects representing all columns in the table field,
 *         or `null` if the field doesn't exist or is not a table field.
 *
 * @throws IllegalArgumentException if the fieldId doesn't correspond to a valid field.
 */
fun DocumentEditor.columnsFor(
    fieldId: String,
    schemaId: String? = null,
): List<Column>? {
    val field = fields.find(fieldId) ?: return null
    return when (field) {
        is TableEditor -> field.columns.all()
        is CollectionEditor -> field.filters.schemas.table(schemaId)?.columns
        else -> null
    }
}