package joyfill.editors.document

import cinematic.Live
import cinematic.MutableLive
import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.Error
import joyfill.ErrorHandler
import joyfill.IdentityGenerator
import joyfill.Stage
import joyfill.View
import joyfill.collections.internal.FieldCollectionImpl
import joyfill.collections.internal.PageCollectionImpl
import joyfill.editors.document.CircularDependencyDetector.CircularDependencyResult
import joyfill.editors.event.EventCapture
import joyfill.editors.resolver.MapBasedJoyDocResolver
import joyfill.editors.utils.indexMap
import joyfill.events.ChangeEvent
import joyfill.tools.validation.FieldsInvalid
import joyfill.tools.validation.FieldsValid
import joyfill.tools.validation.FieldsValidity
import joyfill.tools.validation.Valid
import joyfill.validation.validateSchema
import wisdom.ResolutionResource
import wisdom.ResolutionResourceBuilder
import wisdom.ast.Library
import wisdom.parse
import wisdom.resolutionResource

internal class DocumentEditorImpl(
    validateSchema: Boolean,
    private val document: Document,
    private val identity: IdentityGenerator,
    private val layout: LayoutConfig,
    private val functions: (ResolutionResourceBuilder.() -> Unit)?,
    private val onError: ErrorHandler,
    private val onChange: (DocumentEditor.(ChangeEvent) -> Unit)?,
) : DocumentEditor {

    private val circularDependencyDetector: CircularDependencyDetector by lazy { CircularDependencyDetector(document) }
    private val circularDependencyResult: CircularDependencyResult by lazy { circularDependencyDetector.detectCircularDependencies() }

    override var stage: Stage
        get() = document.stage ?: Stage.draft
        set(value) {
            document.stage = value
        }

    override var name: String
        get() = document.name
        set(value) {
            document.name = value
        }

    override var id: String
        get() = document.id
        set(value) {
            document.id = value
        }

    override var identifier: String
        get() = document.identifier
        set(value) {
            document.identifier = value
        }

    private val resolver by lazy { MapBasedJoyDocResolver(document, circularDependencyResult.circularFields) }

    private val resource: ResolutionResource = resolutionResource {
        include(wisdom.std.core)
        include(wisdom.std.logic)
        include(wisdom.std.math)
        include(wisdom.std.string)
        include(wisdom.std.date)
        include(wisdom.std.arrays)

        getter(resolver::resolve)

        functions?.invoke(this)
    }

    private val library: Library? by lazy {
        val descMap = document.formulas.associateBy({ it.desc }, { it.expression })
        val idMap = document.formulas.associateBy({ it.id }, { it.expression })
        val formulaLookup = descMap + idMap

        val codes = document.fields
            .asSequence()
            .filter { it.id !in circularDependencyResult.circularFields && it.formulas.isNotEmpty() }
            .mapNotNull { field ->
                val formular = field.formulas.first()
                formulaLookup[formular.formula]?.let { expression ->
                    field.id to expression
                }
            }
            .toMap()

        parse(codes)
    }

    override val fields by lazy {
        FieldCollectionImpl(
            document = document,
            pages = pages,
            identity = identity,
            resolver = resource,
            library = library,
            onChange = { onChange?.invoke(this, it) },
            circularFields = circularDependencyResult.circularFields,
        )
    }

    override val pages by lazy { PageCollectionImpl(document, layout) }

    override val capturedEvents: MutableList<EventCapture> = mutableListOf()

    private val _error: MutableLive<Error?> = mutableLiveOf(null)
    override val error: Live<Error?> = _error

    override val views: List<View> by lazy { document.files.flatMap { it.views } }

    override fun set(key: String, value: Any?) = document.set(key, value)

    override fun <R> get(key: String): R = document.get(key)

    override fun toMap() = document.toMap()

    override fun integrity(): FieldsValidity = TODO("Not yet implemented.")

    override fun resolveConditions() = TODO("Not yet implemented.")

    override fun toDocument(): Document = document

    override fun toJsonObject() = document.toJsonObject()

    override fun validate(): FieldsValidity {
        if (error.value != null) return FieldsInvalid(emptyList())

        val them = pages.pages
            .filter { !it.hidden }
            .flatMap { fields.getNonHiddenFieldFrom(it) }
            .associateBy { it.id }

        val sorter = them.keys.indexMap()

        val all = them.values.map { it.validate() }.sortedBy { sorter[it.component.id] }
        val valid = all.filter { it is Valid }.sortedBy { sorter[it.component.id] }

        return if (valid.size == all.size) {
            FieldsValid(valid)
        } else {
            FieldsInvalid(all)
        }
    }

    override fun toJsonString() = document.toJsonString()

    init {
        if (validateSchema) {
            val error = document.validateSchema()
            if (error != null) {
                _error.value = error
                onError.onError(error)
            }
        }
    }
}
