package joyfill.editors.document

import joyfill.Document

/**
 * Detects circular dependencies in document formulas and marks affected fields
 */
internal class CircularDependencyDetector(private val document: Document) {
    
    data class CircularDependencyResult(
        val hasCircularDependencies: Boolean,
        val circularFields: Set<String>,
        val circularChains: List<List<String>>
    )
    
    /**
     * Detects all circular dependencies in the document
     */
    fun detectCircularDependencies(): CircularDependencyResult {
        val dependencyGraph = buildDependencyGraph()
        val state = CycleDetectionState()

        for (fieldId in dependencyGraph.keys) {
            if (fieldId !in state.visited) {
                detectCycles(fieldId, dependencyGraph, state)
            }
        }
        
        return CircularDependencyResult(
            hasCircularDependencies = state.hasCircularDependencies,
            circularFields = state.allCircularFields,
            circularChains = state.circularChains
        )
    }

    /**
     * Builds a dependency graph from field formulas
     */
    private fun buildDependencyGraph(): Map<String, Set<String>> {
        val graph = mutableMapOf<String, MutableSet<String>>()
        
        // Initialize all fields in the graph
        document.fields.forEach { field ->
            graph[field.id] = mutableSetOf()
        }

        val descMap = document.formulas.associateBy({ it.desc }, { it })
        val idMap = document.formulas.associateBy({ it.id }, { it })
        val formulaLookup = descMap + idMap

        // Build dependencies from formulas
        document.fields.forEach { field ->
            if (field.formulas.isNotEmpty()) {
                val formulaApp = field.formulas.first()
                val formula = formulaLookup[formulaApp.formula]
                formula?.let {
                    val dependencies = extractFieldReferences(formula.expression)
                    graph[field.id]?.addAll(dependencies)
                }
            }
        }
        
        return graph
    }


    private val fieldIds: Set<String> by lazy {
        document.fields.mapTo(mutableSetOf()) { it.id }
    }

    private val tableCellPattern = Regex("""([a-zA-Z_][a-zA-Z0-9_]*)\.\d+\.[a-zA-Z_][a-zA-Z0-9_]*""")
    private val identifierRegex = Regex("""[a-zA-Z_][a-zA-Z0-9_]*""")

    /**
     * Extracts field references from a formula expression
     */
    private fun extractFieldReferences(expression: String): Set<String> {
        // First, remove all table cell access patterns to avoid false circular dependencies
        // Table cell patterns: fieldId.number.propertyName (e.g., "table1.0.text1", "table1.5.value")
        val cleanedExpression = expression.replace(tableCellPattern, "")

        return identifierRegex.findAll(cleanedExpression)
            .map { it.value }
            .filter { it in fieldIds }
            .toSet()
    }
    
    /**
     * Detects cycles using DFS with recursion stack tracking
     */
    private fun detectCycles(
        currentNode: String,
        graph: Map<String, Set<String>>,
        state: CycleDetectionState
    ) {
        state.visited.add(currentNode)
        state.recursionStack.add(currentNode)
        state.currentPath.add(currentNode)
        state.pathPositions[currentNode] = state.currentPath.lastIndex

        val dependencies = graph[currentNode] ?: emptySet()
        
        for (dependency in dependencies) {
            if (dependency !in state.visited) {
                detectCycles(dependency, graph, state)
            } else if (dependency in state.recursionStack) {
                // Found a cycle - extract the circular portion
                val cycleStartIndex = state.pathPositions[dependency] ?: -1
                if (cycleStartIndex >= 0) {
                    val cycle = state.currentPath.subList(cycleStartIndex, state.currentPath.size) + listOf(dependency)
                    state.circularChains.add(cycle.toList())
                    state.allCircularFields.addAll(cycle)
                }
            }
        }

        state.currentPath.removeAt(state.currentPath.lastIndex)
        state.pathPositions.remove(currentNode)
        state.recursionStack.remove(currentNode)
    }

    private class CycleDetectionState {
        val visited = mutableSetOf<String>()
        val recursionStack = mutableSetOf<String>()
        val currentPath = mutableListOf<String>()
        val pathPositions = mutableMapOf<String, Int>()
        val circularChains = mutableListOf<List<String>>()
        val allCircularFields = mutableSetOf<String>()
        val hasCircularDependencies get() = allCircularFields.isNotEmpty()
    }
} 