package joyfill.editors.components.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.collections.PageCollection
import joyfill.components.internal.AbstractStringComponent
import joyfill.editors.components.AbstractCompStringEditor
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.notify
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import wisdom.ResolutionResource
import wisdom.ast.Library

internal abstract class AbstractCompStringEditorImpl<T : AbstractStringComponent>(
    override val component: T,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String,
    dispatcher: EventDispatcher?,
) : AbstractComponentEditor(
    document = document,
    pages = pages,
    component = component,
    onValidate = onValidate,
    onChange = onChange,
    identifier = identifier,
    parentFieldId = fieldId,
    dispatcher = dispatcher,
    dependents = dependents,
    resolver = resolver,
    library = library
), AbstractCompStringEditor {

    override val state by lazy {
        val field = component as? joyfill.Field
        
        // Calculate initial value from formula if present
        val initialValue = if (field != null && field.formulas.isNotEmpty() && library != null) {
            try {
                // Use the field ID as the library key (library is indexed by field ID)
                val result = library.call(component.id, resolver)
                // Convert result to string - handle numeric results from functions like length()
                when (result) {
                    is String -> result
                    is Number -> result.toString()
                    is Boolean -> result.toString()
                    null -> null
                    else -> result.toString()
                } ?: ""
            } catch (_: Exception) {
                component.value
            }
        } else {
            component.value
        }
        
        // Update component value directly without triggering state updates during initialization
        if (initialValue != component.value) {
            value(initialValue)
        }

        mutableLiveOf(
            CompStringEditorStateImpl(
                data = initialValue,
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate(),
            )
        )
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override val isValid: Boolean
        get() = component.value?.isNotEmpty() == true

    override fun value(v: String?, origin: Origin) {
        if (component.value?.equals(v, ignoreCase = false) == true) return
        component.value = v
        resolveConditions()
        state.value = state.value.copy(
            data = v,
            visibility = if (hidden) Visibility.Hidden else Visibility.Visible,
            validity = validate(),
        )
        updateDependentsValues()
        if (origin.notify) notifyChange(v)
    }
}