package joyfill.editors.collection.internal

import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collections.PageCollection
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.RowManager
import joyfill.editors.collection.TableEditor
import joyfill.editors.collection.Tables
import joyfill.editors.components.ComponentEditor
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.table.TableSchema
import joyfill.table.internal.TableComponentImpl
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class TablesImpl(
    private val id: String,
    private val host: CollectionEditor,
    private val table: TableSchema,
    private val initialHiddenMap: Map<String, Boolean>,
    private val document: Document,
    private val pages: PageCollection,
    private val onChange: ((ChangeEvent) -> Unit)?,
    private val parent: RowManager,
    private val setRowHidden: (String) -> Unit,
    private val setTableHidden: (String) -> Unit,
    private val identity: IdentityGenerator,
    private val next: () -> RowManager?,
    private val prev: () -> RowManager?,
    private val onAppend: (tableId: String, rowManager: RowManager, index: Int) -> Unit,
    private val resolve: (id: String, type: ResolveType) -> Unit,
    private val dependents: () -> List<ComponentEditor>,
    private val resolver: ResolutionResource,
    private val library: Library? = null,
    private val dispatcher: EventDispatcher? = null,
) : Tables {
    private val cache = mutableMapOf<String, TableEditor>()

    override fun find(key: String?): TableEditor? {
        val childrenSchemas = host.component.schema.all().filter { it.id in table.children || it.title in table.children }
        return childrenSchemas.find { it.id == key || it.title.contentEquals(key, ignoreCase = true) }?.let { tableSchema ->
            getOrCreateTableEditor(tableSchema)
        }
    }

    override fun all(): List<TableEditor> = host.component.schema.all().filter {
        it.id in table.children || it.title in table.children
    }.mapNotNull { tableSchema ->
        getOrCreateTableEditor(tableSchema)
    }

    private fun getOrCreateTableEditor(schema: TableSchema): TableEditor? {
        cache[schema.id]?.let { return it }

        val container = parent.row.children.find(schema.id) ?: return null

        val child = TableComponentImpl(
            container = container,
            schema = schema
        )

        val editor = NestedTableEditor(
            id = "${this.id}:${child.id}",
            host = host,
            document = document,
            pages = pages,
            onChange = onChange,
            parent = parent,
            initialHiddenMap = initialHiddenMap,
            component = child,
            identity = identity,
            next = next,
            prev =  prev,
            onAppend = onAppend,
            setTableHidden = setTableHidden,
            setRowHidden = setRowHidden,
            resolve = resolve,
            dependents = dependents,
            resolver = resolver,
            library = library,
            dispatcher = dispatcher
        )

        cache[schema.id] = editor

        if (host is CollectionEditorImpl) {
            host.tableEditors.add(editor)
        }

        return editor
    }
}