package joyfill.editors.collection.internal

import joyfill.editors.collection.RowManager
import joyfill.editors.collection.entries.LazyEntry
import joyfill.editors.collection.entries.RowEntry
import joyfill.editors.collection.entries.TableEntry
import joyfill.editors.table.TableEditor
import joyfill.tools.validation.Valid

internal fun TableEntry.expanded(
    expandedRowIds: Set<String> = emptySet(),
    allCurrentEntries: List<LazyEntry> = emptyList()
): List<LazyEntry> {
    val displayingRows = table.rows.state.value.displaying()

    val estimatedSize = 1 + displayingRows.size * 2
    val result = ArrayList<LazyEntry>(estimatedSize)

    result.add(this.copy(expanded = true))

    val childEntriesByTable = if (allCurrentEntries.isNotEmpty()) {
        buildChildEntriesLookup(allCurrentEntries)
    } else {
        emptyMap()
    }

    displayingRows.forEachIndexed { idx, rowManager ->
        val wasExpanded = rowManager.row.id in expandedRowIds

        val rowEntry = RowEntry(
            level = level,
            manager = rowManager,
            expanded = wasExpanded,
            hidden = rowManager.state.value.visibility.isHidden(),
            isValid = rowManager.isValid.value,
            table = table,
            number = idx + 1
        )

        result.add(rowEntry)

        if (wasExpanded) {
            processExpandedRow(
                rowManager = rowManager,
                childEntriesByTable = childEntriesByTable,
                allCurrentEntries = allCurrentEntries,
                result = result
            )
        }
    }

    return result
}

private fun TableEntry.processExpandedRow(
    rowManager: RowManager,
    childEntriesByTable: Map<TableEditor, List<LazyEntry>>,
    allCurrentEntries: List<LazyEntry>,
    result: MutableList<LazyEntry>
) {
    val childTables = rowManager.tables.all()

    childTables.forEach { childTable ->
        val childTableEntry = TableEntry(
            level = level + 1,
            childTable,
            expanded = true,
            hidden = childTable.state.value.visibility.isHidden(),
            parent = rowManager.row,
            isValid = table.state.value.validity is Valid
        )

        val childEntries = childEntriesByTable[childTable] ?: emptyList()

        val hasExpandedChildRows = childEntries.any { entry ->
            entry is RowEntry && entry.expanded
        }

        if (hasExpandedChildRows) {
            val childExpandedRowIds = buildExpandedRowIds(childEntries)
            result.addAll(childTableEntry.expanded(childExpandedRowIds, allCurrentEntries))
        } else {
            result.add(childTableEntry)
        }
    }
}

private fun buildChildEntriesLookup(allCurrentEntries: List<LazyEntry>): Map<TableEditor, List<LazyEntry>> {
    val lookup = mutableMapOf<TableEditor, MutableList<LazyEntry>>()

    allCurrentEntries.forEach { entry ->
        val table = entry.table
        lookup.getOrPut(table) { mutableListOf() }.add(entry)
    }

    return lookup
}

private fun buildExpandedRowIds(childEntries: List<LazyEntry>): Set<String> {
    val expandedIds = mutableSetOf<String>()

    childEntries.forEach { entry ->
        if (entry is RowEntry && entry.expanded) {
            expandedIds.add(entry.manager.row.id)
        }
    }

    return expandedIds
}