package joyfill.editors.collection.internal

import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collection.toNestedTableRow
import joyfill.collections.PageCollection
import joyfill.date.DateColumn
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.RowManager
import joyfill.editors.collection.TableEditor
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.Origin
import joyfill.editors.table.RowResult
import joyfill.editors.table.internal.AbstractRows
import joyfill.editors.table.utils.intiCells
import joyfill.editors.utils.localTimeZone
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.table.Row
import joyfill.utils.ID
import joyfill.utils.TIME_ZONE
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class NestedRowsImpl(
    private val host: CollectionEditor,
    private val document: Document,
    private val pages: PageCollection,
    private val table: TableEditor,
    private val identity: IdentityGenerator,
    private val next: () -> RowManager?,
    private val parent: RowManager?,
    private val setRowHidden: (String) -> Unit,
    private val setTableHidden: (String) -> Unit,
    private val initialHiddenMap: Map<String, Boolean>,
    private val prev: () -> RowManager?,
    private val onAppend: (tableId: String, rowManager: RowManager, index: Int) -> Unit,
    private val onCreate: (row: RowManager, index: Int, origin: Origin) -> Unit,
    private val onDelete: (rows: List<RowManager>, origin: Origin) -> Unit,
    private val onMove: (RowResult<Row>, origin: Origin) -> Unit,
    private val onRowChange: ((columnId: String, ChangeEvent, Row) -> Unit)?,
    private val onChange: ((ChangeEvent) -> Unit)?,
    private val resolve: (id: String, type: ResolveType) -> Unit,
    private val dependents: () -> List<ComponentEditor>,
    private val resolver: ResolutionResource,
    private val library: Library? = null,
    private val dispatcher: EventDispatcher? = null,
) : AbstractRows<RowManager>(
    component = table.component,
    tableFilterProvider = { host.filters.current.value },
    rowOrder = table.component.rowOrder
) {
    override fun Row.toManager(): RowManager {
        val uid = "${table.id}:$id"
        return cache.getOrPut(uid) {
            RowManagerImpl(
                id = uid,
                document = document,
                pages = pages,
                host = host,
                table = table.component,
                initialHiddenMap = initialHiddenMap,
                row = this,
                parent = parent,
                columns = table.component.columns,
                identity = identity,
                onRowChange = { columnId, change -> onRowChange?.invoke(columnId, change, this) },
                onChange = onChange,
                next = next,
                prev = prev,
                setRowHidden = setRowHidden,
                setTableHidden = setTableHidden,
                onAppend = onAppend,
                resolve = resolve,
                dependents = dependents,
                resolver = resolver,
                library = library,
                dispatcher = dispatcher,
            )
        }
    }

    override fun create(empty: Boolean, id: String?): RowManager {
        val base = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Row::deleted.name to false,
            Row::cells.name to table.component.intiCells(empty).toMap(),
            Row::children.name to mutableMapOf<String, Any?>()
        )
        if (table.component.columns.any { it is DateColumn }) {
            base[TIME_ZONE] = localTimeZone.id
        }
        val row = base.toNestedTableRow(host.component.schema, table.component)
        return row.toManager()
    }

    override fun append(defaultId: String?, origin: Origin): RowManager {
        val index = table.state.value.rows.size
        val result = super.append(defaultId, origin)
        onCreate(result, index, origin)
        return result
    }

    override fun appendAfter(id: String, defaultId: String?, origin: Origin): RowManager? {
        val result = super.appendAfter(id, defaultId, origin)
        val index = table.state.value.rows.indexOfFirst { it.row.id == id } + 1
        result?.let {
            onCreate(result, index, origin)
        }
        return result
    }

    override fun down(id: String, by: Int, origin: Origin): RowResult<Row> {
        val result = super.down(id, by, origin)
        onMove(result, origin)
        return result
    }

    override fun up(id: String, by: Int, origin: Origin): RowResult<Row> {
        val result = super.up(id, by, origin)
        onMove(result, origin)
        return result
    }

    private fun internalDelete(rowManager: RowManager) {
        rowManager.tables.all().forEach { tableEditor ->
            if (host is CollectionEditorImpl) {
                host.tableEditors.removeAll { it.id == tableEditor.id }
            }
            tableEditor.rows.all().forEach { rowManager ->
                tableEditor.rows.delete(listOf(rowManager.id))
                internalDelete(rowManager)
            }
        }
    }

    override fun delete(keys: List<String>, origin: Origin): List<RowManager> {
        val result = super.delete(keys, origin)

        onDelete(result, origin)

        result.forEach {
            (it as? RowManagerImpl)?.parent?.validate()
        }

        result.forEach { rowManager ->
            internalDelete(rowManager)
        }
        return result
    }
}