package joyfill.editors.collection.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collection.CollectionComponent
import joyfill.collections.PageCollection
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.FilterCollection
import joyfill.editors.collection.RowManager
import joyfill.editors.collection.TableEditor
import joyfill.editors.collection.entries.LazyEntry
import joyfill.editors.collection.entries.RowEntry
import joyfill.editors.collection.entries.TableEntry
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.Selection
import joyfill.editors.utils.Submission
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.table.internal.TableComponentImpl
import joyfill.tools.validation.Valid
import joyfill.tools.visibility.Visibility
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class CollectionEditorImpl(
    override val component: CollectionComponent,
    private val identity: IdentityGenerator,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    private val dependents: () -> List<ComponentEditor>,
    private val resolver: ResolutionResource,
    private val library: Library? = null,
    dispatcher: EventDispatcher? = null,
) : AbstractComponentEditor(
    document = document,
    component = component,
    onValidate = {},
    onChange = onChange,
    identifier = component.identifier,
    pages = pages,
    dependents = dependents,
    resolver = resolver,
    library = library,
    dispatcher = dispatcher,
), CollectionEditor {

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(visibility = if (value) Visibility.Hidden else Visibility.Visible)
    }

    var rootsToShow: Set<String> = emptySet()

    override val filters: FilterCollection by lazy {
        FilterCollectionImpl(
            schemas = component.schema,
            host = this,
            identity = identity,
            beforeFiltering = { targetSchemaId ->
                unselectAll()
                // Reset any previous filter hiding first
                if (targetSchemaId != root().component.identifier)
                    disableFilter()
            },
            afterFiltering = { targetSchemaId, editors, hasSorter ->
                // Cache current entries and convert rootsToShow to Set for O(1) lookup
                val currentEntries = state.value.entries

                // Remove entries that shouldn't be shown instead of hiding them
                val visibleEntries = currentEntries.filter { entry ->
                    when (entry) {
                        is RowEntry -> entry.id in rootsToShow
                        is TableEntry -> true // there will be only one table entry which is the root
                    }
                }

                state.value = state.value.copy(entries = visibleEntries)
                rebuildEntryIndexes()
                expandUntilSchemaFound(targetSchemaId)

                if (targetSchemaId == root().component.identifier) {
                    resetEntriesState()
                } else {
                    if (hasSorter) {
                        // Cache row entries for faster lookup
                        val rowEntryMap = currentEntries.filterIsInstance<RowEntry>()
                            .associateBy { it.manager.id }

                        editors.mapNotNull { it.parent }.forEach { rowManager ->
                            val rowEntry = rowEntryMap[rowManager.id] ?: return@forEach
                            collapse(rowEntry)
                            // Re-find after collapse since state changed
                            val collapsedEntry = state.value.entries.filterIsInstance<RowEntry>()
                                .find { it.id == rowManager.id } ?: return@forEach
                            expand(collapsedEntry)
                        }
                    }
                    hideInvalidPathsToSchema(targetSchemaId)
                }
            },
            disableFilter = {
                unselectAll()
                disableFilter()
            }
        )
    }

    private fun disableFilter() {
        // Clear visibility cache to prevent memory leaks and stale results
        visibilityCache.clear()
        //reset state when filtering is off
        resetEntriesState()
    }

    private fun resetEntriesState() {
        val root = root()
        val tableEntry = TableEntry(
            level = 0,
            expanded = true,
            table = root,
            isValid = state.value.validity is Valid,
            hidden = false,
        )
        val entries = tableEntry.expanded()
        state.value = state.value.copy(entries = entries)
        rebuildEntryIndexes()
    }

    /**
     * Expand rows until we find the target schema.
     * Caches row filtering to avoid scanning entries twice.
     */
    private fun expandUntilSchemaFound(schemaId: String) {
        var foundTargetSchema = false
        var expansionsMade = true
        var cachedRowEntries: List<RowEntry> = emptyList()

        // Keep expanding until we find the target schema or no more expansions are possible
        while (!foundTargetSchema && expansionsMade) {
            val currentEntries = state.value.entries
            expansionsMade = false

            // Cache the row filtering since we use it twice
            cachedRowEntries = currentEntries.filterIsInstance<RowEntry>()

            foundTargetSchema = cachedRowEntries.any { entry ->
                entry.table.component.identifier == schemaId
            }

            if (!foundTargetSchema) {
                // Reuse cached entries instead of filtering again
                for (rowEntry in cachedRowEntries) {
                    if (!rowEntry.expanded && !rowEntry.hidden) {
                        expand(rowEntry)
                        expansionsMade = true
                    }
                }
            }
        }
    }

    /**
     * Hide rows and tables that can't reach the target schema.
     * Pre-groups entries by level to avoid repeated filtering.
     */
    private fun hideInvalidPathsToSchema(targetSchemaId: String) {
        val allEntries = state.value.entries
        val maxLevel = (allEntries.maxOfOrNull { it.level } ?: 1) - 1

        // Group entries by level once instead of filtering repeatedly
        val initialRowsByLevel = mutableMapOf<Int, MutableList<RowEntry>>()
        allEntries.forEach { entry ->
            if (entry is RowEntry && entry.level >= 1 && entry.level <= maxLevel) {
                initialRowsByLevel.getOrPut(entry.level) { mutableListOf() }.add(entry)
            }
        }

        for (level in maxLevel downTo 1) {
            // Refresh state since previous hiding may have changed things
            val currentEntries = state.value.entries
            val currentRowIds = currentEntries.mapTo(mutableSetOf()) { it.id }
            val rowsAtLevel =
                initialRowsByLevel[level]?.filter { it.id in currentRowIds } ?: continue

            for (rowEntry in rowsAtLevel) {
                val canReachVisible = canReachVisibleTargetSchemaRows(rowEntry, targetSchemaId)

                if (!canReachVisible) {
                    val manager = rowEntry.manager
                    // Hide immediately - other checks depend on updated state
                    if (manager.state.value.visibility.isVisible())
                        manager.setHidden(true, HiddenReason.FILTER, true)
                    manager.tables.all().forEach {
                        if (it.state.value.visibility.isVisible())
                            it.setHiddenTo(true, HiddenReason.FILTER, true)
                    }
                }
            }
        }
    }


    // Cache visibility checks since they're expensive and often repeated
    private val visibilityCache = mutableMapOf<String, Boolean>()

    // Entry lookup indexes for O(1) access instead of O(n) scanning
    private val rowEntryIndex = mutableMapOf<String, RowEntry>()  // entry.id -> RowEntry
    private val tableEntryIndex = mutableMapOf<String, TableEntry>()  // entry.id -> TableEntry
    private val managerIdIndex = mutableMapOf<String, RowEntry>()  // manager.id -> RowEntry

    /**
     * Rebuild entry indexes for fast O(1) lookups. Call this after state changes.
     */
    internal fun rebuildEntryIndexes() {
        rowEntryIndex.clear()
        tableEntryIndex.clear()
        managerIdIndex.clear()

        state.value.entries.forEach { entry ->
            when (entry) {
                is RowEntry -> {
                    rowEntryIndex[entry.id] = entry
                    managerIdIndex[entry.manager.id] = entry
                }

                is TableEntry -> {
                    tableEntryIndex[entry.id] = entry
                }
            }
        }
    }

    /**
     * Check if a row can reach visible rows in the target schema.
     * Uses caching + BFS instead of recursion to avoid stack overflow.
     */
    private fun canReachVisibleTargetSchemaRows(
        rowEntry: RowEntry,
        targetSchemaId: String,
    ): Boolean {
        // Create cache key based on row ID and target schema
        val cacheKey = "${rowEntry.manager.id}:$targetSchemaId"

        // Check cache first
        visibilityCache[cacheKey]?.let { return it }

        val result = canReachVisibleTargetSchemaRowsIterative(rowEntry.manager, targetSchemaId)

        // Cache the result
        visibilityCache[cacheKey] = result

        return result
    }

    /**
     * BFS implementation - faster and won't stack overflow
     */
    private fun canReachVisibleTargetSchemaRowsIterative(
        startRowManager: RowManager,
        targetSchemaId: String,
    ): Boolean {
        // Use a queue for breadth-first search
        val queue = mutableListOf<RowManager>()
        val visited = mutableSetOf<String>()

        queue.add(startRowManager)
        visited.add(startRowManager.id)

        while (queue.isNotEmpty()) {
            val currentRowManager = queue.removeAt(0)
            val nestedTables = currentRowManager.tables.all()

            for (table in nestedTables) {
                // Skip hidden tables
                if (table.state.value.visibility.isHidden()) {
                    continue
                }

                // If this is the target schema, check if it has visible rows
                if (table.component.identifier == targetSchemaId) {
                    val displayingRows = table.rows.state.value.displaying()
                    val hasVisibleRows =
                        displayingRows.any { !it.state.value.visibility.isHidden() }

                    if (hasVisibleRows) {
                        return true
                    }
                    continue // Check other tables too
                }

                // Add child rows to queue for further processing
                val tableRows = table.rows.state.value.displaying()
                for (rowManager in tableRows) {
                    if (!rowManager.state.value.visibility.isHidden() &&
                        rowManager.id !in visited
                    ) {
                        queue.add(rowManager)
                        visited.add(rowManager.id)
                    }
                }
            }
        }
        return false
    }

    val tableEditors: MutableList<TableEditor> = mutableListOf()

    private val cached by lazy {
        val schema = component.schema.root() ?: error("Root table not found")
        val initialHiddenMap = component.schema.all().associate { it.id to it.hidden }
        schema.required = schema.required || component.required
        component.required = schema.required
        NestedTableEditor(
            id = schema.id,
            host = this,
            component = TableComponentImpl(component, schema),
            parent = null,
            identity = identity,
            next = { next() },
            prev = { prev() },
            onAppend = ::onAppend,
            resolve = ::resolve,
            setRowHidden = ::setRowHidden,
            setTableHidden = ::setTableHidden,
            document = document,
            pages = pages,
            onChange = onChange,
            initialHiddenMap = initialHiddenMap,
            dependents = dependents,
            resolver = resolver,
            library = library,
            dispatcher = dispatcher,
        )
    }

    private fun resolve(id: String, type: ResolveType) {
        val entry = rowEntryIndex[id]
        entry?.let {
            when (type) {
                ResolveType.Validation -> updateRowValidity(it)
                ResolveType.Condition -> {}
            }
        }
    }

    private fun setRowHidden(id: String) {
        val entry = managerIdIndex[id]
        entry?.let { setHidden(entry) }
    }

    private fun setTableHidden(id: String) {
        val entry = tableEntryIndex[id]
        entry?.let { setHidden(entry) }
    }

    private fun onAppend(tableId: String, rowManager: RowManager, index: Int) {
        insertRowAtPosition(tableId, rowManager, index)

        if (tableId == root().id) {
            state.value = state.value.copy(
                validity = validate()
            )
        }
        updateDependentsValues()
    }

    override fun root(): TableEditor = cached

    override val state by lazy {
        val root = root()
        val validate = validate()
        val tableEntry = TableEntry(
            level = 0,
            expanded = true,
            table = root,
            isValid = validate is Valid,
            hidden = false,
        )
        val entries = tableEntry.expanded()
        mutableLiveOf(
            CollectionEditorStateImpl(
                entries = entries,
                selected = emptyList(),
                validity = validate,
                form = null,
                visibility = if (hidden) Visibility.Hidden else Visibility.Visible,
            )
        )
    }

    init {
        val root = root()
        tableEditors.add(root)
        populateTables(root)
        // Initialize indexes after state is ready
        rebuildEntryIndexes()
    }

    private fun populateTables(table: TableEditor) {
        table.rows.all().forEach { row ->
            row.tables.all().forEach { populateTables(it) }
        }
    }

    override val isValid: Boolean
        get() {
            return root().rows.state.value.displaying().isNotEmpty()
        }

    override fun addAfter() {
        val selected = state.value.selected
        if (selected.size == 1) {
            val selectedRow = selected.first()
            val selectedTableId = selectedRow.table.id
            tableEntryIndex[selectedTableId] ?: return
            selectedRow.table.rows.appendAfter(selectedRow.manager.row.id)
            unselectAll()
            updateDependentsValues()
        }
    }

    override fun moveUp() {
        val selected = state.value.selected
        if (selected.size == 1) {
            val selectedRow = selected.first()
            val selectedTableId = selectedRow.table.id
            val table = tableEntryIndex[selectedTableId] ?: return
            val rowResult = table.table.rows.up(selectedRow.manager.row.id)
            moveRowToIndex(table, rowResult)
            updateDependentsValues() // Notify dependent fields of data change
        }
    }

    override fun moveDown() {
        val selected = state.value.selected
        if (selected.size == 1) {
            val selectedRow = selected.first()
            val selectedTableId = selectedRow.table.id
            val table = tableEntryIndex[selectedTableId] ?: return
            val rowResult = table.table.rows.down(selectedRow.manager.row.id)
            moveRowToIndex(table, rowResult)
            updateDependentsValues() // Notify dependent fields of data change
        }
    }

    override fun addFormAfter() {
        val form = state.value.form ?: return close()
        val formTableId = form.table.id
        tableEntryIndex[formTableId] ?: return close()
        val createdRowManger = form.table.rows.appendAfter(form.manager.row.id)

        val addedRowEntry = rowEntryIndex.values
            .firstOrNull {
                it.manager.row.id == createdRowManger?.row?.id
            } ?: return
        state.value = state.value.copy(
            form = addedRowEntry
        )
        updateDependentsValues() // Notify dependent fields of data change
    }

    override fun submit(update: (Submission) -> Unit) {
        val form = state.value.form ?: return close()
        val selected = state.value.selected
        var count = 0
        val total = selected.size
        for (row in selected) {
            update(Submission.Submitting(count, total))
            form.manager.copyInto(row.manager)
            count++
        }
        update(Submission.Submitted(total))
        close()
        updateDependentsValues() // Notify dependent fields of data change
    }

    override fun toggle(entry: LazyEntry) = when (entry) {
        is RowEntry -> if (entry.expanded) collapse(entry) else expand(entry)
        is TableEntry -> if (entry.expanded) collapse(entry) else expand(entry)
    }

    override fun expandRow(entry: RowEntry) {
        expand(entry)
    }

    override val isMulti: Boolean
        get() = filters.current.value?.state?.value?.isActive?.not() ?: true

    override fun select(item: RowEntry?, origin: Origin): RowEntry? {
        item ?: return null

        if (!isMulti || item.table.id != state.value.selected.firstOrNull()?.table?.id) unselectAll()
        val selected = state.value.selected
        if (selected.any { it.id == item.id }) return null
        state.value = state.value.copy(selected = selected + item)
        return item
    }

    override fun unselect(item: RowEntry?, origin: Origin): RowEntry? {
        item ?: return null

        val selected = state.value.selected
        if (selected.none { it.id == item.id }) return null

        state.value = state.value.copy(selected = selected.filter { it.id != item.id })

        return item
    }

    override fun unselectAll(origin: Origin) {
        state.value = state.value.copy(selected = emptyList())
    }

    override fun edit() {
        val selected = state.value.selected
        if (selected.isEmpty()) return close()
        val first = selected.first()
        if (selected.size > 1) {
            val newRowManager = first.table.rows.create(true)
            val newRowEntry = first.copy(manager = newRowManager)
            state.value = state.value.copy(form = newRowEntry)
        } else {
            state.value = state.value.copy(form = first)
        }
    }

    override fun selectAll(entry: TableEntry) {
        unselectAll()
        val selectedRowEntryList = state.value.entries.filterIsInstance<RowEntry>()
            .filter { it.table.id == entry.table.id }
        state.value = state.value.copy(
            selected = selectedRowEntryList
        )
    }

        override fun delete() {
        val candidates = state.value.selected
        if (candidates.isEmpty()) return

        val selectedTable = candidates.first().table
        
        // Capture the returned deleted managers instead of ignoring them
        val deletedManagers = selectedTable.rows.delete(candidates.map { it.manager.row.id })

        // Use the optimized deletion method with the returned data
        val modifiedEntries = removeRows(deletedManagers)

        val tableEntry = tableEntryIndex.values.firstOrNull { it.table.id == selectedTable.id }
        tableEntry?.let {
            updateRowNumbers(modifiedEntries, tableEntry.table.id)
            
            // Find and update the table entry in the modified entries
            val tableIndex = modifiedEntries.indexOfFirst { 
                it.id == tableEntry.id && it is TableEntry 
            }
            
            if (tableIndex != -1) {
                val updatedTableEntry = tableEntry.copy(
                    isValid = tableEntry.table.state.value.validity is Valid
                )
                modifiedEntries[tableIndex] = updatedTableEntry
            }
        }
        
        state.value = state.value.copy(
            entries = modifiedEntries,
            selected = emptyList()
        )
        if (selectedTable.id == root().id){
            state.value = state.value.copy(
                validity = validate()
            )
        }
        rebuildEntryIndexes()
        updateDependentsValues()
    }

    override fun close() {
        state.value = state.value.copy(form = null)
    }

    private fun prev(): RowManager? {
        val currentForm = state.value.form ?: return null
        val tableRows = state.value.entries.filterIsInstance<RowEntry>()
            .filter { it.table.id == currentForm.table.id }
        val prevNumber = currentForm.number - 1
        val entry = tableRows.firstOrNull { it.number == prevNumber } ?: return null
        state.value = state.value.copy(form = entry)
        return entry.manager
    }

    private fun next(): RowManager? {
        val currentForm = state.value.form ?: return null
        val tableRows = state.value.entries.filterIsInstance<RowEntry>()
            .filter { it.table.id == currentForm.table.id }
        val nextNumber = currentForm.number + 1
        val entry = tableRows.firstOrNull { it.number == nextNumber } ?: return null
        state.value = state.value.copy(form = entry)
        return entry.manager
    }

    override fun selection(entry: TableEntry): Selection {
        val selected = state.value.selected.filter { entry.table.id == it.table.id }
        val rows = entry.table.rows.state.value.displaying()
        return when {
            selected.isEmpty() -> Selection.None
            selected.size == rows.size -> Selection.All
            selected.isEmpty() -> Selection.None
            selected.isNotEmpty() -> Selection.Some(selected.map { it.manager })
            else -> Selection.None
        }
    }
}