package joyfill.editors.chart.internal

import cinematic.MutableLive
import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.chart.ChartField
import joyfill.chart.Line
import joyfill.editors.chart.LineCollection
import joyfill.editors.chart.LineEditor
import joyfill.editors.event.EventTrigger
import joyfill.events.ChangeEvent

@PublishedApi
internal class LineCollectionImpl(
    document: Document,
    val identity: IdentityGenerator,
    val field: ChartField,
    onChange: ((ChangeEvent) -> Unit)?,
    private val onValueChange: (() -> Unit)? = null
) : LineCollection,
    EventTrigger<ChartField>(
        document = document,
        component = field,
        fieldId = field.id,
        fieldIdentifier = field.identifier,
        onChange = onChange
    )
{
    override fun all(): List<LineEditor> = field.value.map { editor(it) }

    private fun look(key: String): Line? = field.value.find { it.id == key || it.title == key }

    override val all: MutableLive<List<LineEditor>> by lazy { mutableLiveOf(all()) }

    override fun find(key: String): LineEditor? {
        val line = look(key) ?: return null
        return editor(line)
    }

    private val cache = mutableMapOf<String, LineEditor>()
    private fun editor(line: Line) = cache.getOrPut(line.id) {
        LineEditorImpl(
            identity = identity,
            lineId = line.id,
            getLine = { field.value.find { it.id == line.id } },
            onValueChange = {
                onValueChange?.invoke()
            }
        )
    }

    private fun value(newValue: MutableList<Line>?) {
        // Clear cache when replacing all lines
        if (newValue != field.value) {
            cache.clear()
        }

        // Update field value - clear and replace with new values
        field.value.clear()
        field.value.addAll(newValue.orEmpty())

        // Update current live value
        all.value = all()

        // Notify about the change
        notifyChange(newValue?.map { it.toMap() }?.toMutableList())

        // Trigger validation and dependent updates
        onValueChange?.invoke()
    }

    override fun add(
        title: String,
        id: String?,
        description: String?
    ): LineEditor {
        val line = Line(
            id = id ?: identity.generate(),
            title = title,
            description = description,
        )

        // Create new value with added line - don't modify field.value directly
        val newValue = field.value.toMutableList().apply { add(line) }

        val e = editor(line)
        value(newValue)
        return e
    }

    override fun replaceAll(lines: MutableList<Line>?) {
        value(lines)
    }

    override fun remove(key: String): Line? {
        val line = look(key) ?: return null

        // Create new value with removed line - don't modify field.value directly
        val newValue = field.value.toMutableList().apply { remove(line) }

        value(newValue)
        return line
    }
}