package joyfill.editors.chart.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.chart.Axis
import joyfill.chart.ChartField
import joyfill.chart.Line
import joyfill.chart.toLine
import joyfill.collections.PageCollection
import joyfill.editors.chart.ChartEditor
import joyfill.editors.chart.LineCollection
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.table.HiddenReason
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.tools.validation.ComponentInvalid
import joyfill.tools.validation.ComponentValid
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import joyfill.utils.ID
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class ChartEditorImpl(
    override val component: ChartField,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library? = null,
    document: Document,
    private val identity: IdentityGenerator,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit = {},
    dispatcher: EventDispatcher? = null,
) : AbstractComponentEditor(
    document = document,
    pages = pages,
    component = component,
    onValidate = onValidate,
    onChange = onChange,
    identifier = component.identifier,
    dispatcher = dispatcher,
    dependents = dependents,
    resolver = resolver,
    library = library
), ChartEditor {

    override val lines: LineCollection = LineCollectionImpl(
        document = document,
        identity = identity,
        field = component,
        onChange = onChange,
        onValueChange = {
            // Update state validity when lines change
            state.value = state.value.copy(
                validity = validate()
            )

            resolveConditions()
            // Update dependent values when lines change
            updateDependentsValues()
        }
    )

    override val state by lazy {
        mutableLiveOf(
            ChartEditorStateImpl(
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate(),
            )
        )
    }

    override val axes by lazy {
        AxesEditorImpl(
            x = Axis(
                label = component.xTitle.orEmpty(),
                min = component.xMin ?: 0.0,
                max = component.xMax ?: 100.0
            ),
            y = Axis(
                label = component.yTitle.orEmpty(),
                min = component.yMin ?: 0.0,
                max = component.xMax ?: 100.0
            )
        )
    }

    fun value(newValue: MutableList<Line>?) {
        // Use the centralized value method from LineCollection
        // This will handle field update, validation, and dependent updates
        lines.replaceAll(newValue)
        // Also notify change at the chart level
        resolveConditions()
        // Update dependent values when lines change
        updateDependentsValues()
        notifyChange(newValue)
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible
        )
    }

    override fun validate(): ComponentValidity {
        return if (component.required && (component.value.isEmpty() || component.value.all { it.points.isEmpty() })) {
            ComponentInvalid(
                component = component,
                messages = listOf("Component ${component.title} is required")
            )
        } else {
            ComponentValid.empty(component)
        }
    }

    @Suppress("UNCHECKED_CAST")
    internal fun linesFrom(result: Any?): MutableList<Line>? = when {
        result is List<*> && result.all { it is Line } -> result.toMutableList() as MutableList<Line>
        result is List<*> -> {
            result.mapNotNull { item ->
                when (item) {
                    is Line -> item
                    is Map<*, *> -> {
                        val lineAsMap = item as MutableMap<String, Any?>

                        val points = lineAsMap[Line::points.name] as? List<*>
                        val convertedPoints: MutableList<MutableMap<String, Any?>>? = points?.map { point ->
                            val pointsMap = point as? MutableMap<String, Any?> ?: mutableMapOf()
                            if (!pointsMap.containsKey(ID)) pointsMap[ID] = identity.generate()
                            pointsMap
                        }?.toMutableList()
                        lineAsMap[Line::points.name] = convertedPoints
                        if (!lineAsMap.containsKey(ID)) lineAsMap[ID] = identity.generate()
                        lineAsMap.toLine()
                    }

                    else -> null
                }
            }.toMutableList()
        }

        else -> null
    }

    init {
        val initialValue = if (component.formulas.isNotEmpty()) {
            val result = library?.call(component.id, resolver)
            linesFrom(result)
        } else {
            null
        }

        if (initialValue != null) value(initialValue)
    }
} 