package joyfill.collections.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.FieldPosition
import joyfill.IdentityGenerator
import joyfill.Page
import joyfill.collections.PageCollection
import joyfill.collections.PageCollectionState
import joyfill.conditions.AndEval
import joyfill.conditions.HideAction
import joyfill.conditions.Logic
import joyfill.conditions.OrEval
import joyfill.conditions.ShowAction
import joyfill.conditions.field.FieldConditionDefinition
import joyfill.conditions.field.FieldLogic
import joyfill.editors.document.LayoutConfig
import joyfill.editors.page.PageEditor
import joyfill.editors.page.PageEditorImpl
import joyfill.toField
import joyfill.toPage
import joyfill.utils.ID
import joyfill.utils.POSITIONS

internal class PageCollectionImpl(
    private val document: Document,
    private val layout: LayoutConfig,
    private val page: String? = null
): PageCollection {
    private val files by lazy { document.files }

    private val views by lazy { files.flatMap { it.views } }

    private val view by lazy { views.find { it.type == layout.preferred.name.lowercase() } }

    val pages by lazy { view?.pages ?: files.flatMap { it.pages } }

    override val state by lazy { mutableLiveOf(computePageState(page)) }

    private fun computePageState(key: String?): PageCollectionState {
        val them = pages.mapNotNull { find(it.id) }.filter { !it.page.hidden }.map { it.page }
        val current = find(key)?.page ?: them.firstOrNull()
        return PageCollectionState(view, them, current)
    }

    override fun raw(): List<Page> = files.flatMap { it.pages }

    override fun all() = files.flatMap { it.pages }.map {
        cache.getOrPut(it.id) { PageEditorImpl(document, it) }
    }

    private val cache = mutableMapOf<String, PageEditor>()
    override fun find(key: String?): PageEditor? {
        val k = key ?: return null
        val page = pages.find {
            it.id == k || it.name == k || it.identifier == k
        } ?: return null
        return cache.getOrPut(page.id) { PageEditorImpl(document, page) }
    }

    override fun at(index: Int): PageEditor? {
        if (index < 0) return null
        val pages = files.flatMap { it.pages }
        if (index >= pages.size) return null
        val page = pages[index]
        return cache.getOrPut(page.id) { PageEditorImpl(document, page) }
    }

    override fun setHidden(key: String?, value: Boolean): Page? {
        val editor = find(key) ?: return null
        val p = editor.page
        editor.page.hidden = value
        state.value = computePageState(state.value.page?.id)
        return p
    }

    override fun navigate(page: Page?): Page? {
        val p = page ?: return null
        state.value = state.value.copy(page = p)
        return p
    }

    override fun navigate(page: String?): Page? = navigate(find(page)?.page)

    override fun duplicate(page: String?, newName: String): Page? {
        val p = find(page)?.page
        return duplicate(p, newName)
    }

    override fun duplicate(page:Page?, newName: String): Page? {
        if (page == null) return null

        val mapped = page.toMap().toMutableMap()
        val id = IdentityGenerator.default.generate()
        mapped[ID] = id
        mapped[Page::name.name] = newName
        //regenerate field positions ids
        mapped[POSITIONS] =  (mapped[POSITIONS] as List<Map<String, Any>>? ?: emptyList() ).map {
            it.toMutableMap().apply { this[ID] = IdentityGenerator.default.generate() }
        }

        val fieldInPositions  = page.positions.map { it.field }.toSet()
        val oldToNewFieldsIds = mutableMapOf<String, String>().apply {
            fieldInPositions.forEach { put(it, IdentityGenerator.default.generate()) }
        }

        val fields = document.fields
            .filter { it.id in fieldInPositions }
            .map { f ->
                val mappedField = f.toMap().toMutableMap()
                //update this field id in the positions
                mapped[POSITIONS] = (mapped[POSITIONS] as List<Map<String, Any>>? ?: emptyList() ).map {
                    if (it[FieldPosition::field.name] == f.id) {
                        it.toMutableMap().apply { this[FieldPosition::field.name] = oldToNewFieldsIds[f.id] as Any }
                    } else {
                        it
                    }
                }

                //add to doc fields
                mappedField[ID] = oldToNewFieldsIds[f.id]
                if(f.logic != null){
                    val conditions = f.logic!!.conditions.map {
                        if (it.field in oldToNewFieldsIds.keys) {
                            val condMap = it.toMap().toMutableMap()
                            condMap[FieldConditionDefinition::field.name] = oldToNewFieldsIds[it.field]
                            condMap
                        } else {
                            it.toMap()
                        }
                    }
                    mappedField[Page::logic.name] = mutableMapOf<String, Any?>().apply {
                        put(Page::hidden.name, f.logic?.hidden)
                        put(Logic::action.name, when(f.logic?.action){
                            ShowAction -> "show"
                            HideAction -> "hide"
                            else -> "unknown"
                        })
                        put(Logic::eval.name, when(f.logic?.eval){
                            AndEval -> "and"
                            OrEval -> "or"
                            else -> "unknown"
                        })

                        put(FieldLogic::conditions.name, conditions)
                    }
                }
                mappedField.toField()
            }
        document.fields.addAll(fields)
        document.files.first().apply {
            val p = mapped.toPage()
            pages.add(p)
            pageOrder.add(p.id)
        }

        val newPage = mapped.toPage()
        state.value = state.value.copy(pages = state.value.pages + newPage, page = newPage)
        return newPage
    }
}