package joyfill.editors.table.internal

import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collections.PageCollection
import joyfill.editors.collection.internal.TableFilterImpl
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.Columns
import joyfill.editors.table.Origin
import joyfill.editors.table.RowManager
import joyfill.editors.table.RowResult
import joyfill.editors.table.TableEditor
import joyfill.editors.table.notify
import joyfill.editors.utils.localTimeZone
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.events.Target
import joyfill.table.Cell
import joyfill.table.Row
import joyfill.table.TableComponent
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class TableEditorImpl(
    override val component: TableComponent,
    private val identity: IdentityGenerator,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
    dispatcher: EventDispatcher?,
) : TableEditor, AbstractTableEditor(
    component = component,
    initialHidden = component.hidden,
    document = document,
    pages = pages,
    onChange = onChange,
    identifier = component.identifier,
    fieldId = component.id,
    dispatcher = dispatcher,
    dependents = dependents,
    resolver = resolver,
    library = library
) {

    override val columns: Columns by lazy { ColumnsImpl(component.columns) }

    override val filters by lazy {
        TableFilterImpl(
            schema = component,
            getEditors = { listOf(this) },
            identity = identity,
            beforeFiltering = {
                unselectAll()
            },
            root = this,
        )
    }

    override fun getCell(column: String): Cell? = null

    override val rows by lazy {
        RowsImpl(
            component = component,
            document = document,
            identity = identity,
            filter = filters,
            onAppend = ::onAppendRow,
            onDelete = ::onDelete,
            onAddAfter = ::onAddAfter,
            onMove = ::onMove,
            onChange = ::onRowChange,
            next = ::next,
            prev = ::prev,
            dependents = dependents,
            resolver = resolver,
            library = library,
            dispatcher = dispatcher,
        )
    }

    override fun resolveConditions() {}

    private fun onRowChange(columnId: String, changeEvent: ChangeEvent, row: Row) {
        if (row.timeZone == null && row.cells.date(columnId) != null) row.timeZone = localTimeZone.id
        changeEvent.changelogs.forEach { changeLog ->
            val value = buildRowUpdateChangeValue(columnId, row, changeLog)
            notifyChange(value, Target.field_value_rowUpdate)
        }
        resolveConditions()
    }

    init {
        state.value = state.value.copy(
            validity = validate()
        )
    }

    private fun onAppendRow(manager: RowManager, origin: Origin) {
        if (manager.row.id in state.value.rows.map { it.row.id }) return

        if (origin.notify) {
            val value = buildRowCreateChangeValue(manager, state.value.rows.size)
            notifyChange(
                value = value,
                target = Target.field_value_rowCreate
            )
        }
        val newRows = state.value.rows + manager
        state.value = state.value.copy(
            rows = newRows,
            validity = validate(newRows)
        )
        updateDependentsValues()
    }

    private fun onMove(rowResult: RowResult<Row>, origin: Origin) {
        if (rowResult.index < 0) return

        if (origin.notify) {
            val value = buildRowMoveChangeValue(rowResult.row, rowResult.index)
            notifyChange(
                value = value,
                target = Target.field_value_rowMove
            )
        }

        val newRows = component.value.mapNotNull { row ->
            state.value.rows.find { it.row.id == row.id }
        }

        state.value = state.value.copy(rows = newRows)
        updateDependentsValues()
    }

    private fun onDelete(keys: List<String>, origin: Origin) {
        val removed = state.value.rows.filter { it.row.id in keys }
        val retained = state.value.rows - removed.toSet()

        if (origin.notify) {
            removed.forEach {
                val value = buildRowDeleteChangeValue(it)
                notifyChange(
                    value = value,
                    target = Target.field_value_rowDelete
                )
            }
        }

        state.value = state.value.copy(
            rows = retained,
            validity = validate(retained)
        )
        unselectAll()
        updateDependentsValues()
    }

    private fun onAddAfter(id: String, manager: RowManager?, origin: Origin) {
        if (manager == null) return
        val index = state.value.rows.indexOfFirst { it.row.id == id }
        if (index != -1) {
            val updatedRows = state.value.rows.toMutableList()
            updatedRows.add(index + 1, manager)

            if (origin.notify) {
                val value = buildRowCreateChangeValue(manager, index + 1)
                notifyChange(
                    value = value,
                    target = Target.field_value_rowCreate
                )
            }

            state.value = state.value.copy(rows = updatedRows)
            unselectAll()
        }
        updateDependentsValues()
    }
}