package joyfill.editors.table.internal

import joyfill.conditions.Action
import joyfill.conditions.AndEval
import joyfill.conditions.Contains
import joyfill.conditions.Empty
import joyfill.conditions.Equals
import joyfill.conditions.Filled
import joyfill.conditions.GreaterThan
import joyfill.conditions.HideAction
import joyfill.conditions.LessThan
import joyfill.conditions.NotEquals
import joyfill.conditions.OrEval
import joyfill.conditions.ShowAction
import joyfill.conditions.UnknownConditionOperator
import joyfill.conditions.UnknownEval
import joyfill.conditions.table.TableConditionDefinition
import joyfill.conditions.table.TableLogic
import joyfill.date.DateCell
import joyfill.dropdown.DropdownCell
import joyfill.editors.components.ValueEvaluator
import joyfill.multi_select.MultiSelectCell
import joyfill.number.NumberCell
import joyfill.table.Cell
import joyfill.value.ListValue
import joyfill.value.SingleValue
import kotlinx.datetime.LocalDate

internal class TableConditionEvaluator(private val getCell: (String) -> Cell?): ValueEvaluator {
    
    fun evaluateConditions(
        logic: TableLogic,
        currentHidden: Boolean,
        initialHidden: Boolean,
    ): Boolean {
        val conditions = logic.conditions
        if (conditions.isEmpty()) return currentHidden

        return when (logic.eval) {
            AndEval -> isActionMet(logic.action, initialHidden){ conditions.all { isConditionMet(it) } }
            OrEval -> isActionMet(logic.action, initialHidden){ conditions.any { isConditionMet(it) } }
            is UnknownEval -> initialHidden
        }
    }

    private fun isActionMet(
        action: Action,
        initialHidden: Boolean,
        conditionMet: () -> Boolean,
    ): Boolean {
        return if (conditionMet() && action is ShowAction) {
            false
        } else if (conditionMet() && action is HideAction) {
            true
        } else {
            initialHidden
        }
    }

    private fun isConditionMet(condition: TableConditionDefinition): Boolean {
        val cell = getCell(condition.column) ?: return false
        val v = cell.value

        return when (condition.condition) {
            Filled -> cell.isFilled()
            Contains -> cell.hasValueContaining(condition.value)
            Empty -> isValueEmpty(v)
            Equals -> cell.hasValueEqualTo(condition.value)
            NotEquals -> !cell.hasValueEqualTo(condition.value)
            GreaterThan -> isGreaterThan(v, condition.value)
            LessThan -> isLessThan(v, condition.value)
            is UnknownConditionOperator -> false
        }
    }

    private fun Cell.hasValueContaining(value: Any?): Boolean {
        val rhs = value
        return when (this) {
            is MultiSelectCell -> {
                val lhs = column.options.filter { it.id in this.value }
                return when (rhs) {
                    is String -> lhs.find { it.id == rhs || it.value == rhs } != null
                    is Array<*> -> rhs.all { v -> v in lhs.map { it.id } || v in lhs.map { it.value } }
                    null -> lhs.isEmpty()
                    else -> false
                }
            }

            is DropdownCell -> hasValueEqualTo(value)

            else -> this.value?.toString()?.contains(rhs.toString(), ignoreCase = true) == true
        }
    }

    private fun Cell.hasValueEqualTo(rhs: Any?): Boolean {
        if (value == null && rhs == null) return true
        if (value != null && rhs == null) return false
        return when (this) {
            is MultiSelectCell -> {
                val lhs = column.options.filter { it.id in this.value }
                return when (rhs) {
                    is String -> lhs.find { it.id == rhs || it.value == rhs } != null
                    is Array<*> -> rhs.all { v -> v in lhs.map { it.id } || v in lhs.map { it.value } }
                    null -> lhs.isEmpty()
                    else -> false
                }
            }

            is DropdownCell -> {
                val lhs = column.options.firstOrNull { it.id == value }
                return when (rhs) {
                    is String -> lhs?.id == rhs || lhs?.value == rhs
                    is Array<*> -> rhs.size == 1 && (lhs?.id == rhs.first() || lhs?.value == rhs.first())
                    null -> lhs == null
                    else -> false
                }
            }

            is NumberCell -> {
                value == rhs?.toString()?.toDoubleOrNull()
            }

            is DateCell -> when (rhs) {
                is String -> value == LocalDate.parse(rhs).toEpochDays().toLong()
                is Number -> value == rhs
                else -> false
            }

            else -> when (val v = value) {
                is String -> v.contentEquals("$rhs", ignoreCase = true)
                else -> v == rhs
            }
        }
    }

    private fun Cell.isFilled(): Boolean = when (this) {
        is ListValue<*> -> value.isNotEmpty()
        is SingleValue<*> -> when (val v = value) {
            is String -> v.isNotBlank()
            else -> v != null
        }

        else -> false
    }
} 