package joyfill.editors.table.internal

import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.date.DateColumn
import joyfill.editors.collection.TableFilter
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.Origin
import joyfill.editors.table.RowManager
import joyfill.editors.table.RowResult
import joyfill.editors.table.utils.intiCells
import joyfill.editors.utils.localTimeZone
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.table.Row
import joyfill.table.TableComponent
import joyfill.table.toTableRow
import joyfill.utils.ID
import joyfill.utils.TIME_ZONE
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class RowsImpl(
    private val component: TableComponent,
    private val document: Document,
    private val identity: IdentityGenerator,
    private val filter: TableFilter?,
    private val onAppend: (RowManager, Origin) -> Unit,
    private val onDelete: (List<String>, Origin) -> Unit,
    private val onAddAfter: (String, RowManager?, Origin) -> Unit,
    private val onMove: (RowResult<Row>, Origin) -> Unit,
    private val onChange: ((columnId: String, ChangeEvent, Row) -> Unit)?,
    private val next: () -> RowManager?,
    private val prev: () -> RowManager?,
    private val dependents: () -> List<ComponentEditor>,
    private val resolver: ResolutionResource,
    private val library: Library? = null,
    private val dispatcher: EventDispatcher? = null,
) : AbstractRows<RowManager>(
    component = component,
    tableFilterProvider = { filter },
    rowOrder = component.rowOrder
) {

    val required by lazy {
        component.columns.any { it.required }
    }

    override fun create(empty: Boolean, id: String?): RowManager {
        val base = mutableMapOf<String, Any?>(
            ID to (id ?: identity.generate()),
            Row::deleted.name to false,
            Row::cells.name to component.intiCells(empty).toMap()
        )
        if (component.columns.any { it is DateColumn }) {
            base[TIME_ZONE] = localTimeZone.id
        }

        val row = base.toTableRow(component.columns)

        val manager = RowManagerImpl(
            row = row,
            columns = component.columns,
            required = required,
            document = document,
            identity = identity,
            onChange = { columnId, change -> onChange?.invoke(columnId, change, row) },
            next = next,
            prev = prev,
            fieldId = component.id,
            identifier = component.identifier,
            dependents = dependents,
            resolver = resolver,
            library = library,
            dispatcher = dispatcher
        )
        return manager
    }

    override fun append(defaultId: String?, origin: Origin): RowManager {
        val manager = super.append(defaultId, origin)
        onAppend(manager, origin)
        return manager
    }

    override fun appendAfter(id: String, defaultId: String?, origin: Origin): RowManager? {
        val manager = super.appendAfter(id, defaultId, origin)
        onAddAfter(id, manager, origin)
        return manager
    }

    override fun delete(keys: List<String>, origin: Origin): List<RowManager> {
        val result = super.delete(keys, origin)
        onDelete(keys, origin)
        return result
    }

    override fun down(id: String, by: Int, origin: Origin): RowResult<Row> {
        val result = super.down(id, by, origin)
        onMove(result, origin)
        return result
    }

    override fun up(id: String, by: Int, origin: Origin): RowResult<Row> {
        val result = super.up(id, by, origin)
        onMove(result, origin)
        return result
    }

    override fun find(id: String): RowManager? = cache.getOrPut(id) {
        val row = component.value.find { it.id == id } ?: return null
        row.toManager()
    }

    override fun Row.toManager() = RowManagerImpl(
        row = this,
        columns = component.columns,
        document = document,
        identity = identity,
        onChange = { columnId, change -> onChange?.invoke(columnId, change, this) },
        next = next,
        required = required,
        prev = prev,
        fieldId = component.id,
        identifier = component.identifier,
        dependents = dependents,
        resolver = resolver,
        library = library,
        dispatcher = dispatcher
    )
}