package joyfill.editors.table.internal

import joyfill.barcode.BarcodeColumn
import joyfill.block.BlockColumn
import joyfill.date.DateColumn
import joyfill.dropdown.DropdownColumn
import joyfill.editors.table.RowManager
import joyfill.editors.utils.localTimeZone
import joyfill.editors.utils.timezoneFromId
import joyfill.image.ImageColumn
import joyfill.multi_select.MultiSelectColumn
import joyfill.number.NumberColumn
import joyfill.signature.SignatureColumn
import joyfill.table.Column
import joyfill.text.TextColumn
import kotlinx.datetime.Instant
import kotlinx.datetime.toInstant
import kotlinx.datetime.toLocalDateTime

internal fun RowManager.copyInto(other: RowManager, column: Column) {
    when (column) {
        is TextColumn -> {
            val value = text(column.id)?.state?.value?.data
            if (value.isNullOrBlank()) return
            other.text(column.id)?.value(value) ?: return
        }

        is DateColumn -> {
            val value = date(column.id)?.state?.value?.data ?: return
            if (value == 0L) return

            val adjusted = if (row.timeZone == other.row.timeZone) {
                value
            } else {
                val srcTz = timezoneFromId(row.timeZone) ?: localTimeZone
                val dstTz = timezoneFromId(other.row.timeZone) ?: localTimeZone
                val local = Instant.fromEpochMilliseconds(value).toLocalDateTime(srcTz)
                local.toInstant(dstTz).toEpochMilliseconds()
            }
            other.date(column.id)?.value(adjusted)
        }

        is BlockColumn -> {
            val value = block(column.id)?.state?.value?.data
            if (value.isNullOrBlank()) return
            other.block(column.id)?.value(value)
        }

        is NumberColumn -> {
            val value = number(column.id)?.state?.value?.data ?: return
            if (value.isNaN() || value == 0.0) return
            other.number(column.id)?.value(value)
        }

        is DropdownColumn -> {
            val dropdown = dropdown(column.id) ?: return
            val selected = dropdown.state.value.data ?: return
            val cell = other.dropdown(column.id) ?: return
            if (cell.component.value == dropdown.component.value) return
            cell.select(selected)
        }

        is MultiSelectColumn -> {
            val multiSelect = multiSelect(column.id) ?: return
            val selected = multiSelect.state.value.data
            if (selected.isEmpty()) return
            val cell = other.multiSelect(column.id) ?: return
            if (cell.component.value == multiSelect.component.value) return
            cell.set(selected)
        }

        is ImageColumn -> {
            val image = image(column.id) ?: return
            val images = image.state.value.data
            if (images.isEmpty()) return
            val cell = other.image(column.id) ?: return
            if (cell.component.value == image.component.value) return
            cell.set(images)
        }

        is BarcodeColumn -> {
            val value = barcode(column.id)?.state?.value?.data
            if (value.isNullOrBlank()) return
            other.barcode(column.id)?.value(value) ?: return
        }

        is SignatureColumn -> {
            val value = signature(column.id)?.state?.value?.data
            if (value.isNullOrBlank()) return
            other.signature(column.id)?.value(value) ?: return
        }

        else -> {}
    }
}
