package joyfill.editors.table.internal

import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.barcode.BarcodeCell
import joyfill.block.BlockCell
import joyfill.date.DateCell
import joyfill.dropdown.DropdownCell
import joyfill.editors.barcode.BarcodeEditorImpl
import joyfill.editors.block.BlockEditorImpl
import joyfill.editors.components.ComponentEditor
import joyfill.editors.date.internal.DateEditorImpl
import joyfill.editors.dropdown.internal.DropdownEditorImpl
import joyfill.editors.file.internal.FileEditorImpl
import joyfill.editors.image.ImageEditorImpl
import joyfill.editors.multi_select.internal.MultiSelectEditorImpl
import joyfill.editors.number.internal.NumberEditorImpl
import joyfill.editors.signature.SignatureEditorImpl
import joyfill.editors.text.TextEditorImpl
import joyfill.editors.text_area.TextAreaEditorImpl
import joyfill.editors.unknown.UnknownEditor
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.file.FileCell
import joyfill.image.ImageCell
import joyfill.multi_select.MultiSelectCell
import joyfill.number.NumberCell
import joyfill.signature.SignatureCell
import joyfill.table.Cell
import joyfill.table.Column
import joyfill.text.TextCell
import joyfill.text_area.TextAreaCell
import joyfill.tools.validation.ComponentValidity
import wisdom.ResolutionResource
import wisdom.ast.Library

internal fun Cell.toEditor(
    cache: MutableMap<String, ComponentEditor>,
    validation: MutableMap<Column, ComponentValidity>,
    document: Document,
    identity: IdentityGenerator,
    onChange: ((ChangeEvent) -> Unit)?,
    fieldId: String,
    identifier: String,
    onCellValidated: (Cell, ComponentValidity) -> Unit,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library? = null,
    dispatcher: EventDispatcher? = null,
): ComponentEditor = cache.getOrPut(id) {
    val editor = when (this) {
        is TextCell -> TextEditorImpl(
            component = this,
            document = document,
            pages = null,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            dependents = dependents,
            resolver = resolver,
            library = library,
            dispatcher = dispatcher
        )

        is BlockCell -> BlockEditorImpl(
            component = this,
            document = document,
            onChange = onChange,
            onValidate = { onCellValidated(this, it) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = null,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is TextAreaCell -> TextAreaEditorImpl(
            component = this,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is SignatureCell -> SignatureEditorImpl(
            component = this,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is DropdownCell -> DropdownEditorImpl(
            component = this,
            options = column.options,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library,
        )

        is NumberCell -> NumberEditorImpl(
            component = this,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is DateCell -> DateEditorImpl(
            component = this,
            format = column.format,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            library = library,
            resolver = resolver,
            dependents = dependents
        )

        is BarcodeCell -> BarcodeEditorImpl(
            component = this,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is ImageCell -> ImageEditorImpl(
            component = this,
            identity = identity,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is FileCell -> FileEditorImpl(
            component = this,
            identity = identity,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        is MultiSelectCell -> MultiSelectEditorImpl(
            component = this,
            options = column.options,
            document = document,
            onChange = onChange,
            onValidate = { validity -> onCellValidated(this, validity) },
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )

        else -> UnknownEditor(
            component = this,
            document = document,
            onChange = null,
            fieldId = fieldId,
            identifier = identifier,
            pages = null,
            dispatcher = dispatcher,
            dependents = dependents,
            resolver = resolver,
            library = library
        )
    }
    validation[this.column] = editor.state.value.validity
    editor
}