package joyfill.editors.table.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.collections.PageCollection
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.RowManager
import joyfill.editors.table.RowVisibility
import joyfill.editors.table.Selection
import joyfill.editors.table.TableEditor
import joyfill.editors.utils.Submission
import joyfill.events.ChangeEvent
import joyfill.events.ChangeLog
import joyfill.events.EventDispatcher
import joyfill.table.Cell
import joyfill.table.Row
import joyfill.table.TableComponent
import joyfill.tools.validation.ComponentInvalid
import joyfill.tools.validation.ComponentValid
import joyfill.tools.validation.ComponentValidity

import wisdom.ResolutionResource
import wisdom.ast.Library

internal abstract class AbstractTableEditor(
    override val component: TableComponent,
    initialHidden: Boolean,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    identifier: String,
    fieldId: String,
    dispatcher: EventDispatcher?,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
) : AbstractComponentEditor(document, pages, component, {}, onChange, identifier, fieldId, dispatcher, dependents, resolver, library), TableEditor {

    private val changeLogBuilder by lazy { TableChangeLogBuilder(component.columns) }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value

        rows.state.value.all.forEach {
            it.setHidden(value, reason, liveUpdate)
        }

        state.value = state.value.copy(
            visibility = if (value) RowVisibility.Hidden(reason) else RowVisibility.Visible
        )
    }

    override val state by lazy {
        mutableLiveOf(
            TableEditorStateImpl(
                validity = validate(),
                rows = rows.state.value.all,
                selected = emptyList(),
                form = null,
                visibility =  if (initialHidden) RowVisibility.Hidden(HiddenReason.CONDITIONAL_LOGIC) else RowVisibility.Visible,
            )
        )
    }

    abstract fun getCell(column: String): Cell?

    internal fun validate(rows: List<RowManager>): ComponentValidity {
        return when {
            component.required && rows.isEmpty() ->
                ComponentInvalid(component, listOf("Component ${component.title} is required"))
            else -> ComponentValid.empty(component)
        }
    }

    override fun validate(): ComponentValidity {
        return validate(rows.state.value.displaying())
    }

    override val isMulti: Boolean
        get() = !filters.state.value.isActive

    override fun select(item: RowManager?, origin: Origin): RowManager? {
        item ?: return null
        if (item in state.value.selected) return item

        if (!isMulti) unselectAll()

        val selectedItem = item.select()

        state.value = state.value.copy(
            selected = state.value.selected + item,
            rows = state.value.rows.map {
                if (it.id == selectedItem.id) selectedItem else it
            }
        )

        return selectedItem
    }

    override fun unselect(item: RowManager?, origin: Origin): RowManager? {
        item ?: return null
        val unselectedRow = item.unselect()

        state.value = state.value.copy(
            selected = state.value.selected - item,
            rows = state.value.rows.map {
                if (it.id == unselectedRow.id) unselectedRow else it
            }
        )

        return unselectedRow
    }

    override fun selectAll() {
        state.value = state.value.copy(
            selected = state.value.rows,
            rows = state.value.rows.map { it.select() }
        )
    }

    override fun unselectAll(origin: Origin) {
        state.value = state.value.copy(
            rows = state.value.rows.map { it.unselect() },
            selected = emptyList()
        )
    }

    override fun edit() {
        val selected = state.value.selected
        if (selected.isEmpty()) return close()
        if (selected.size > 1) {
            val newRowManager = rows.create(true)
            state.value = state.value.copy(form = newRowManager)
        } else {
            state.value = state.value.copy(form = selected.first())
        }
    }

    override fun submit(update: (Submission) -> Unit) {
        val form = state.value.form ?: return close()
        val selected = state.value.selected
        var count = 0
        val total = selected.size
        for (it in selected) {
            update(Submission.Submitting(count, total))
            form.copyInto(it)
            count++
        }
        update(Submission.Submitted(total))
        close()
    }

    override fun close() {
        state.value = state.value.copy(
            form = null
        )
    }

    protected fun next(): RowManager? {
        val currentForm = state.value.form
        val nextIndex = state.value.rows.indexOfFirst { it.row.id == currentForm?.row?.id } + 1
        val nextForm = state.value.rows.getOrNull(nextIndex) ?: return null
        state.value = state.value.copy(form = nextForm)
        return nextForm
    }

    protected fun prev(): RowManager? {
        val currentForm = state.value.form
        val prevIndex = state.value.rows.indexOfFirst { it.row.id == currentForm?.row?.id } - 1
        val prevForm = state.value.rows.getOrNull(prevIndex) ?: return null
        state.value = state.value.copy(form = prevForm)
        return prevForm
    }

    override fun selection(): Selection = state.value.selection()

    protected fun buildRowCreateChangeValue(rowManager: RowManager, index: Int, includeChildren: Boolean = false) =
        changeLogBuilder.buildRowCreateChangeValue(rowManager, index, includeChildren)

    protected fun buildRowMoveChangeValue(row: Row?, index: Int) =
        changeLogBuilder.buildRowMoveChangeValue(row, index)

    protected fun buildRowDeleteChangeValue(rowManager: RowManager) =
        changeLogBuilder.buildRowDeleteChangeValue(rowManager)

    protected fun buildRowUpdateChangeValue(columnId: String, row: Row, changeLog: ChangeLog) =
        changeLogBuilder.buildRowUpdateChangeValue(columnId, row, changeLog)
}