package joyfill.editors.table.internal

import cinematic.mutableLiveOf
import joyfill.editors.collection.TableFilter
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.RowManager
import joyfill.editors.table.RowResult
import joyfill.editors.table.RowState
import joyfill.editors.table.RowVisibility
import joyfill.editors.table.Rows
import joyfill.editors.table.TableEditor
import joyfill.editors.table.utils.applyFilters
import joyfill.editors.table.utils.sort
import joyfill.table.Row
import joyfill.value.ListValue

internal abstract class AbstractRows<M : RowManager>(
    private val component: ListValue<Row>,
    private val tableFilterProvider: () -> TableFilter?,
    private val rowOrder: MutableList<String>,
) : Rows<M> {
    protected val cache by lazy { mutableMapOf<String, M>() }

    override val state by lazy {
        mutableLiveOf(
            RowState(
                selected = emptyList(),
                filtered = null,
                all = all()
            )
        )
    }

    override fun disableFilters() {
        state.value = state.value.copy(
            selected = emptyList(),
            filtered = null,
        )
    }

    override fun applyFilters(editor: TableEditor): List<M> {
        if (state.value.all.isEmpty()) return state.value.all
        val tableFilter = tableFilterProvider()
        if (tableFilter == null || !tableFilter.state.value.isActive) return state.value.all
        val filterState = tableFilter.state.value
        val sorter = filterState.sorter

        val filtered = state.value.all.applyFilters(filterState.filters).let { filtered ->
            if (sorter != null) filtered.sort(sorter) else filtered
        }.associateBy { it.id }

        for (row in state.value.all) {
            val visibility = row.state.value.visibility

            val isVisibleOrFilterHidden = visibility is RowVisibility.Visible ||
                    (visibility is RowVisibility.Hidden && visibility.reason == HiddenReason.FILTER)

            if (isVisibleOrFilterHidden) {
                row.setHidden(
                    value = row.id !in filtered,
                    reason = HiddenReason.FILTER,
                    liveUpdate = false
                )
            }
        }

        state.value = state.value.copy(
            selected = emptyList(),
            filtered = filtered.values.toList(),
        )

        return state.value.filtered.orEmpty()
    }

    abstract override fun create(empty: Boolean, id: String?): M

    override fun append(defaultId: String?, origin: Origin): M {
        val manager = create(empty = false, id = defaultId)
        component.value.add(manager.row)
        rowOrder.add(manager.row.id)
        cache[manager.row.id] = manager
        state.value = state.value.copy(
            all = state.value.all + manager
        )
        return manager
    }

    override fun appendAfter(id: String, defaultId: String?, origin: Origin): M? {
        val index = state.value.all.indexOfFirst { it.row.id == id }
        if (index == -1) return null
        val manager = create(empty = false, id = defaultId)
        component.value.add(index + 1, manager.row)
        rowOrder.add(index + 1, manager.row.id)
        cache[manager.row.id] = manager
        state.value = state.value.copy(all = all())
        return manager
    }

    override fun find(id: String): M? = cache.getOrPut(id) {
        val row = component.value.find {
            it.id == id && !it.deleted && it.id in rowOrder
        } ?: return null
        row.toManager()
    }

    override fun all(): List<M> = component.value.filter {
        !it.deleted && it.id in rowOrder
    }.map {
        cache.getOrPut(it.id) { it.toManager() }
    }

    override fun delete(keys: List<String>, origin: Origin): List<M> {
        val rows = component.value.toList()
        val stateRows = state.value.all
        val removed = rows.filter { it.id in keys }
        val removedManagers = removed.mapNotNull { row ->
            stateRows.find { it.row.id == row.id }
        }
        val retained = rows - removed.toSet()

        cache.keys.removeAll { it in keys }
        component.value.clear()
        component.value.addAll(retained)
        state.value = state.value.copy(all = all())
        rowOrder.clear()
        rowOrder.addAll(retained.filter { !it.deleted }.map { it.id })
        return removedManagers
    }

    abstract fun Row.toManager(): M

    private fun moveRow(id: String, indexOffset: Int): RowResult<Row> {
        if (indexOffset == 0) return RowResult(null, -1)

        val currentRows = component.value.toList()
        val rowIndex = currentRows.indexOfFirst { it.id == id }
        if (rowIndex == -1) return RowResult(null, -1)

        val row = currentRows[rowIndex]

        val targetIndex = (rowIndex + indexOffset).coerceIn(0, currentRows.lastIndex)

        if (rowIndex == targetIndex) return RowResult(row, rowIndex)

        val newRows = currentRows.toMutableList().apply {
            removeAt(rowIndex)
            add(targetIndex, row)
        }

        component.value.clear()
        component.value.addAll(newRows)
        rowOrder.clear()
        rowOrder.addAll(newRows.map { it.id })
        state.value = state.value.copy(all = all())
        return RowResult(row, targetIndex)
    }

    override fun up(id: String, by: Int, origin: Origin): RowResult<Row> = moveRow(id, -by)

    override fun down(id: String, by: Int, origin: Origin): RowResult<Row> = moveRow(id, by)
}


internal fun getRootRowId(string: String): String {
    val parts = string.split(":")
    return parts.take(2).joinToString(":")
}
