package joyfill.editors.table.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditorFinder
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.RowManager
import joyfill.editors.table.RowVisibility
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.table.Cell
import joyfill.table.Column
import joyfill.table.Row
import joyfill.tools.validation.ComponentInvalid
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.validation.Valid
import wisdom.ResolutionResource
import wisdom.ast.Library

internal abstract class AbstractRowManager(
    override val row: Row,
    private val document: Document,
    val columns: List<Column>,
    val required: Boolean,
    private val identity: IdentityGenerator,
    val onChange: ((columnId: String, ChangeEvent) -> Unit)?,
    val fieldId: String,
    val identifier: String,
    private val dependents: () -> List<ComponentEditor>,
    private val resolver: ResolutionResource,
    private val library: Library?,
    private val dispatcher: EventDispatcher? = null,
) : AbstractComponentEditorFinder(), RowManager {

    override fun copyInto(other: RowManager) {
        for (column in columns) copyInto(other, column)
    }

    override val state by lazy {
        mutableLiveOf(
            RowManagerStateImpl(
                allCellsValid = true,
                isSelected = false,
                visibility = RowVisibility.Visible,
                filterOrder = -1
            )
        )
    }

    override fun setHidden(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        state.value = state.value.copy(
            visibility = if (value) RowVisibility.Hidden(reason) else RowVisibility.Visible
        )
    }

    override fun select(): RowManager = apply {
        state.value = state.value.copy(
            isSelected = true
        )
    }

    override fun unselect(): RowManager = apply {
        state.value = state.value.copy(
            isSelected = false
        )
    }

    protected val cache = mutableMapOf<String, ComponentEditor>()
    private val validation = mutableMapOf<Column, ComponentValidity>()

    private fun Cell.toEditor(): ComponentEditor = toEditor(
        cache = cache,
        validation = validation,
        document = document,
        identity = identity,
        onChange = { onChange?.invoke(column.id, it) },
        fieldId = fieldId,
        identifier = identifier,
        onCellValidated = ::onCellValidated,
        dependents = dependents,
        resolver = resolver,
        library = library,
        dispatcher = dispatcher
    )

    override fun validateRow(): Boolean {
        if (!required) return true
        val result = all().all { it.validate() is Valid }
        state.value = state.value.copy(allCellsValid = result)
        return result
    }

    override fun find(key: String?): ComponentEditor? {
        if (key.isNullOrEmpty()) return null
        return cache.getOrPut(key) { row.cells.find(key)?.toEditor() ?: return null }
    }

    //change
    override fun all(): List<ComponentEditor> = row.cells.all().map { cell ->
        cache.getOrPut(cell.column.id) { cell.toEditor() }
    }

    private fun onCellValidated(cell: Cell, validity: ComponentValidity) {
        validation[cell.column] = validity
        validate()
    }

    private fun validate() {
        if (validation.size != row.cells.all().size) return

        val invalid = validation.any { it.value is ComponentInvalid }
        state.value = state.value.copy(allCellsValid = !invalid)
    }
}
