package joyfill.editors.resolver

import joyfill.Document
import joyfill.block.BlockComponent
import joyfill.chart.ChartField
import joyfill.collection.CollectionField
import joyfill.collection.TableSchemas
import joyfill.components.Component
import joyfill.date.DateColumn
import joyfill.date.DateComponent
import joyfill.date.DateField
import joyfill.dropdown.DropdownColumn
import joyfill.dropdown.DropdownField
import joyfill.multi_select.MultiSelectColumn
import joyfill.multi_select.MultiSelectField
import joyfill.number.NumberColumn
import joyfill.number.NumberComponent
import joyfill.number.NumberField
import joyfill.table.Column
import joyfill.table.Row
import joyfill.table.TableField
import joyfill.table.TableSchema
import joyfill.text.TextColumn
import joyfill.text.TextComponent
import joyfill.text.TextField
import joyfill.text_area.TextAreaComponent
import joyfill.utils.Option
import wisdom.NonExistentField

/**
 * Map-based resolver for Joy documents that provides field values and structured data access for formulas
 */
class MapBasedJoyDocResolver(
    private val document: Document,
    private val circularFields: Set<String> = emptySet(),
) : JoyDocResolver {
    private fun Component.default() = when (this) {
        is NumberComponent -> value ?: 0.0
        is TextComponent, is TextAreaComponent, is BlockComponent -> ""
        is DateComponent -> value ?: 0L
        else -> null
    }

    /**
     * Resolves dropdown value
     */
    private fun resolveDropdownValue(id: String?, options: List<Option>): String? {
        if (id == null) return null
        return options.find { it.id == id && !it.deleted }?.value ?: id
    }

    /**
     * Resolves multiselect ID list to display value list using options
     */
    private fun resolveMultiSelectValues(ids: List<String>?, options: List<Option>): List<String> {
        if (ids == null) return emptyList()
        return options.mapNotNull { if (it.id in ids && !it.deleted) it.value else null }
    }

    /**
     * Converts a collection row to the resolver format for formula access
     */
    private fun convertRowToResolverFormat(
        row: Row,
        schemas: TableSchemas,
        table: TableSchema? = null,
    ): Map<String, Any?> {
        val targetTable = table ?: schemas.root() ?: return emptyMap()
        val rowData = mutableMapOf<String, Any?>()

        // Add all column values - directly access the raw cell values from the row's map
        targetTable.columns.forEach { column ->
            val cell = row.cells.find(column.id)
            val cellValue = cell?.value ?: cell?.default()
            rowData[column.id] = convertCellValueForFormula(cellValue, column)
            rowData[column.title.lowercase()] = convertCellValueForFormula(cellValue, column)
        }

        // Add nested children
        if (row.children.all().isNotEmpty()) {
            val childrenData = mutableMapOf<String, Any?>()
            targetTable.children.forEach { childSchemaId ->
                val childCollection = row.children.find(childSchemaId)
                if (childCollection != null) {
                    val childTable = schemas.table(childSchemaId)
                    if (childTable != null) {
                        childrenData[childSchemaId] = childCollection.value.map { childRow ->
                            convertRowToResolverFormat(childRow, schemas, childTable)
                        }
                    }
                }
            }
            if (childrenData.isNotEmpty()) {
                rowData["children"] = childrenData
            }
        }

        return rowData
    }

    /**
     * Converts cell values for formula evaluation, handling ID to display value conversion
     */
    private fun convertCellValueForFormula(cellValue: Any?, column: Column): Any? {
        return when (column) {
            is DropdownColumn -> {
                resolveDropdownValue(cellValue as? String, column.options)
            }

            is MultiSelectColumn -> {
                resolveMultiSelectValues(cellValue as? List<String>, column.options)
            }

            is NumberColumn -> {
                // Convert number values to proper numeric format
                when (cellValue) {
                    is Number -> cellValue.toDouble()
                    is String -> cellValue.toDoubleOrNull()
                    else -> null
                }
            }

            is DateColumn -> {
                // For date columns, return the raw value which should be a LocalDate
                // The wisdom engine will handle date functions like day(), month(), etc.
                // However, treat timestamp 0 as null since it represents an empty date field
                when (cellValue) {
                    0L -> null  // Empty date field should be null, not Jan 1, 1970
                    is Number -> if (cellValue.toLong() == 0L) null else cellValue
                    else -> cellValue
                }
            }

            is TextColumn -> {
                // For text columns, ensure we have a string value
                cellValue?.toString()
            }

            else -> {
                cellValue
            }
        }
    }

    private fun resolvePropertyAccess(propertyPath: String): Any? {
        val parts = propertyPath.split(".")
        if (parts.size < 2) return null // Need at least "field.property"

        val fieldId = parts[0]
        val field = document.fields.find { it.id == fieldId } ?: return null

        // Handle simple field property access (e.g., "text.value", "text.disabled")
        if (parts.size == 2) {
            val propertyName = parts[1]
            return when (propertyName) {
                "value" -> when (field) {
                    is TextField -> field.value
                    is NumberField -> field.value
                    is DateField -> field.value
                    is DropdownField -> resolveDropdownValue(field.value, field.options)
                    is MultiSelectField -> resolveMultiSelectValues(field.value, field.options)
                    else -> field.value
                }

                "disabled" -> field.disabled
                "hidden" -> field.hidden
                "required" -> field.required
                "title" -> field.title
                "id" -> field.id
                "identifier" -> field.identifier
                else -> null
            }
        }

        // Handle complex field property access (3+ parts)
        if (parts.size < 3) return null

        return when (field) {
            is CollectionField -> {
                val rootSchema = field.schema.root() ?: return null
                resolveRowCell(field.value, rootSchema.columns, parts)
            }

            is TableField -> {
                // Handle table property access (e.g., table1.0.text1)
                resolveRowCell(field.value, field.columns, parts)
            }

            is ChartField -> {
                // Handle chart property access (e.g., chart1.1.points.0.y)
                try {
                    when (parts.size) {
                        2 -> {
                            // chart1.1 -> return specific line
                            val lineIndex = parts[1].toInt()
                            val value = field.value
                            if (lineIndex >= value.size) return null
                            val line = value[lineIndex]
                            return line.toMap()
                        }

                        3 -> {
                            // chart1.1.points -> return all points from line
                            val lineIndex = parts[1].toInt()
                            val property = parts[2]
                            val value = field.value
                            if (lineIndex >= value.size) return null
                            val line = value[lineIndex]

                            return when (property) {
                                "points" -> line.points.map { point -> point.toMap() }
                                "title" -> line.title
                                "description" -> line.description
                                else -> null
                            }
                        }

                        4 -> {
                            // chart1.1.points.0 -> return specific point
                            val lineIndex = parts[1].toInt()
                            val pointIndex = parts[3].toInt()
                            val value = field.value
                            if (lineIndex >= value.size) return null
                            val line = value[lineIndex]
                            if (pointIndex >= line.points.size) return null
                            val point = line.points[pointIndex]

                            return point.toMap()
                        }

                        5 -> {
                            // chart1.1.points.0.y -> return specific point property
                            val lineIndex = parts[1].toInt()
                            val pointIndex = parts[3].toInt()
                            val pointProperty = parts[4]

                            val value = field.value
                            if (lineIndex >= value.size) return null
                            val line = value[lineIndex]
                            if (pointIndex >= line.points.size) return null
                            val point = line.points[pointIndex]

                            return when (pointProperty) {
                                "x" -> point.x
                                "y" -> point.y
                                "label" -> point.label
                                "_id" -> point.id
                                else -> null
                            }
                        }

                        else -> null
                    }
                } catch (_: NumberFormatException) {
                    null
                }
            }

            else -> null
        }
    }

    private fun resolveRowCell(
        rows: List<Row>,
        columns: List<Column>,
        parts: List<String>,
    ): Any? {
        val rowIndex = parts[1].toIntOrNull() ?: return null
        val propertyName = parts[2]

        val row = rows.getOrNull(rowIndex) ?: return null
        val column = columns.find {
            it.id == propertyName || it.title.equals(propertyName, ignoreCase = true)
        } ?: return null

        val cell = row.cells.find(column.id)
        val cellValue = cell?.value ?: cell?.default()
        return convertCellValueForFormula(cellValue, column)
    }

    override fun resolve(id: String): Any? {
        // Return null for any field involved in circular dependencies
        val fieldId = if (id.contains(".")) id.split(".").first() else id
        if (fieldId in circularFields) return null

        if (!id.contains(".")) {
            return when (val field = document.fields.find { it.id == id }) {
                is TextField -> field.value
                is NumberField -> field.value ?: 0.0
                is DateField -> field.value
                is DropdownField -> resolveDropdownValue(field.value, field.options)
                is MultiSelectField -> resolveMultiSelectValues(field.value, field.options)
                is CollectionField -> convertCollectionToStructuredData(field)
                is TableField -> convertTableToStructuredData(field)
                is ChartField -> field.value.map { line -> line.toMap() }
                null -> NonExistentField
                else -> field.value
            }
        }

        // Handle property access like "table1.0.text1"
        return resolvePropertyAccess(id)
    }

    private fun convertCollectionToStructuredData(field: CollectionField): List<Map<String, Any?>> {
        val result = mutableListOf<Map<String, Any?>>()
        if (field.value.isEmpty()) return result
        for (row in field.value) {
            val rowData = convertRowToResolverFormat(row, field.schema, null)
            if (rowData.isNotEmpty()) result.add(rowData)
        }
        return result
    }

    private fun convertTableToStructuredData(field: TableField): List<Map<String, Any?>> {
        val result = mutableListOf<Map<String, Any?>>()
        if (field.value.isEmpty()) return result
        for (row in field.value) {
            val rowMap = mutableMapOf<String, Any?>()
            for (column in field.columns) {
                val cell = row.cells.find(column.id)
                val cellValue = cell?.value ?: cell?.default()
                // Use both column ID and lowercase title as keys for accessibility
                rowMap[column.id] = convertCellValueForFormula(cellValue, column)
                rowMap[column.title.lowercase()] = convertCellValueForFormula(cellValue, column)
            }
            if (rowMap.isNotEmpty()) result.add(rowMap)
        }
        return result
    }
}