package joyfill.editors.number.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.collections.PageCollection
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.number.NumberEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.notify
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.number.NumberComponent
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class NumberEditorImpl(
    override val component: NumberComponent,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String = component.id,
    dispatcher: EventDispatcher?,
) : NumberEditor, AbstractComponentEditor(
    document,
    pages,
    component,
    onValidate,
    onChange,
    identifier,
    fieldId,
    dispatcher,
    dependents,
    resolver,
    library
) {

    override val state by lazy {
        val field = component as? joyfill.Field

        // Calculate initial value from formula if present
        val initialValue = if (field != null && field.formulas.isNotEmpty() && library != null) {
            try {
                val result = library.call(component.id, resolver)
                when (result) {
                    is Number -> result.toDouble()
                    is String -> result.toDoubleOrNull()
                    else -> null
                } ?: 0.0
            } catch (_: Exception) {
                component.value
            }
        } else {
            component.value
        }

        // Update component value directly without triggering state updates during initialization
        if (initialValue != component.value) {
            value(initialValue)
        }

        mutableLiveOf(
            NumberEditorStateImpl(
                data = initialValue,
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate()
            )
        )
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override fun value(v: Double?, origin: Origin) {
        if (component.value == v) return
        component.value = v
        resolveConditions()
        state.value = state.value.copy(
            data = v,
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        updateDependentsValues()
        if (origin.notify) notifyChange(v)
    }

    override val isValid: Boolean
        get() = component.value != null
}