package joyfill.editors.event

import joyfill.Document
import joyfill.components.Component
import joyfill.events.ChangeEvent
import joyfill.events.ChangeLog
import joyfill.events.Target
import joyfill.events.internal.FieldUpdateChangeImpl
import joyfill.events.toChangeEvent
import joyfill.table.Column
import joyfill.table.TableSchema
import joyfill.utils.ID
import joyfill.utils.TIME_ZONE

internal open class EventTrigger<out C : Component>(
    private val document: Document,
    open val component: C,
    private val fieldId: String,
    private val fieldIdentifier: String,
    private val onChange: ((ChangeEvent) -> Unit)?
) {
    val file by lazy { document.files[0] }

    val page by lazy {
        document.files.flatMap { it.pages }.find { page ->
            val positions = page.positions.map { it.field }
            positions.contains(fieldId)
        } ?: throw IllegalStateException("Component not found in any page")
    }

    val position by lazy {
        page.positions
            .distinctBy { it.field }
            .find { it.field == fieldId }
            ?: throw IllegalStateException("Component not found in any position")
    }

    protected fun notifyChange(
        value: Any?,
        target: String = Target.field_update
    ) {
        onChange?.invoke((createChangeEvent(EventCapture(value = value, target = target))))
    }

    private fun createChangeEvent(capture: EventCapture): ChangeEvent = mutableMapOf<String, Any?>(
        ChangeEvent::changelogs.name to mutableListOf(createChangeLog(capture)),
        ChangeEvent::document.name to document.toMap(),
    ).toChangeEvent(
        columns = ((capture.value as? MutableMap<String, Any?>)?.get(TableSchema::columns.name) as? List<Column>)
            ?: emptyList()
    )

    private fun createChangeLog(capture: EventCapture): MutableMap<String, Any> {
        val isNotFieldUpdate = capture.target != Target.field_update
        return mutableMapOf(
            ChangeLog::sdk.name to "kotlin",
            ChangeLog::v.name to 1,
            ChangeLog::target.name to capture.target,

            ID to document.id,
            ChangeLog::identifier.name to document.identifier,

            ChangeLog::fileId.name to file.id,
            ChangeLog::pageId.name to page.id,
            ChangeLog::fieldPositionId.name to position.id,
            ChangeLog::fieldId.name to if (isNotFieldUpdate) fieldId else component.id,
            ChangeLog::fieldIdentifier.name to fieldIdentifier,

            ChangeLog::change.name to when (capture.target) {
                Target.field_update -> {
                    if (capture.value is MutableMap<*, *>) {
                        mutableMapOf(
                            FieldUpdateChangeImpl::value.name to capture.value[FieldUpdateChangeImpl::value.name],
                            TIME_ZONE to capture.value[TIME_ZONE],
                        )
                    } else {
                        mutableMapOf(FieldUpdateChangeImpl::value.name to capture.value)
                    }
                }
                Target.field_value_rowCreate,
                Target.field_value_rowUpdate,
                Target.field_value_rowDelete,
                Target.field_value_rowMove -> (capture.value as MutableMap<String, Any?>).toMutableMap()
                    .apply { remove(TableSchema::columns.name) }

                else -> mutableMapOf(FieldUpdateChangeImpl::value.name to capture.value)
            },

            ChangeLog::createdOn.name to capture.timestamp,
        )
    }
}