package joyfill.editors.dropdown.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.collections.PageCollection
import joyfill.dropdown.DropdownComponent
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.dropdown.DropdownEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.notify
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import joyfill.utils.Option
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class DropdownEditorImpl(
    override val component: DropdownComponent,
    override val options: List<Option>,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String = component.id,
    dispatcher: EventDispatcher?,
) : AbstractComponentEditor(
    document = document,
    pages = pages,
    component = component,
    onValidate = onValidate,
    onChange = onChange,
    identifier = identifier,
    parentFieldId = fieldId,
    dispatcher = dispatcher,
    dependents = dependents,
    resolver = resolver,
    library = library
), DropdownEditor {

    override val state by lazy {
        val field = component as? joyfill.Field

        val initialValue = if (field != null && field.formulas.isNotEmpty()) {
            library?.call(field.id, resolver) as? String
        } else {
            component.value
        }
        value(initialValue)
        mutableLiveOf(
            DropdownEditorStateImpl(
                data = options.firstOrNull { it.value == component.value || it.id == component.value },
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate()
            )
        )
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override fun value(v: String?, origin: Origin) {
        val option = options.firstOrNull { it.value == v || it.id == v }

        if (component.value == option?.id) return

        // Set the component value first
        component.value = option?.id
        // Find the corresponding option for the UI state
        // Update conditions and state
        resolveConditions()
        state.value = state.value.copy(
            data = option,
            validity = validate(),
            visibility = if (hidden) Visibility.Hidden else Visibility.Visible,
        )
        // Update dependents and notify
        updateDependentsValues()
        if (origin.notify) notifyChange(option?.id)
    }

    private fun look(key: String?): Option? =
        options.firstOrNull { it.id == key || it.value == key }

    override fun select(key: String?, origin: Origin) {
        val option = look(key) ?: return
        if (component.value == option.value) return
        value(option.value, origin)
    }

    override fun select(item: Option?, origin: Origin): Option? {
        select(item?.value, origin)
        return item
    }

    override fun unselect(item: Option?, origin: Origin): Option? {
        unselect(item?.value, origin)
        return item
    }

    override fun unselect(key: String?, origin: Origin) {
        val option = look(key) ?: return
        if (component.value == option.value || component.value == option.id) {
            value(null, origin)
        }
    }

    override val isValid: Boolean
        get() = component.value?.isNotEmpty() == true

    override fun unselectAll(origin: Origin) {
        value(null, origin)
    }
}