package joyfill.editors.document.helper

import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.entries.TableEntry
import joyfill.editors.collection.internal.CollectionEditorImpl
import joyfill.editors.collection.internal.expand
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.TableEditor
import joyfill.editors.collection.TableEditor as CollectionTableEditor

internal fun ComponentEditor.tableEditor(
    rowId: String? = null,
    schemaId: String? = null,
    parentPath: String? = null,
): TableEditor? = when (this) {
    is TableEditor -> this
    is CollectionEditor -> nestedEditor(rowId, schemaId, parentPath)
    else -> null
}

internal fun CollectionEditor.nestedEditor(
    rowId: String?,
    schemaId: String?,
    parentPath: String?,
): TableEditor {
    if (this !is CollectionEditorImpl) return root()

    val targetSchemaTableEditors = tableEditors.filter { it.identifier == schemaId }

    if (rowId != null) {
        return targetSchemaTableEditors.find { it.rows.find(rowId) != null } ?: root()
    }

    if (!parentPath.isNullOrEmpty() && '.' in parentPath) {
        return root().getByParentPath(parentPath) ?: root()
    }

    return targetSchemaTableEditors.firstOrNull() ?: root()
}

internal fun CollectionTableEditor.getByParentPath(parentPath: String): CollectionTableEditor? {
    val parts = parentPath.split('.').chunked(2)
    var table: CollectionTableEditor = this
    parts.forEach { (index, schemaId) ->
        val row = table.rows.state.value.all.getOrNull(index.toInt())
        table = row?.tables?.find(schemaId) ?: return this
    }
    return table
}

internal fun ComponentEditor.refresh(tableEditor: TableEditor) {
    // For collection fields, we need to update the collection editor's state
    // to reflect the row move in the UI
    val field = this
    if (field is CollectionEditorImpl) {
        // Find the table entry that contains the moved row and refresh its state
        val tableEntry = field.state.value.entries
            .filterIsInstance<TableEntry>()
            .find { it.table.id == tableEditor.id }

        if (tableEntry is TableEntry) {
            field.expand(tableEntry)
        }
    }
}
