package joyfill.editors.document.helper

import joyfill.editors.document.DocumentEditor
import joyfill.events.ChangeLog
import joyfill.events.RowCreateChange
import joyfill.events.RowDeleteChange
import joyfill.events.RowMoveChange
import joyfill.events.RowUpdateChange
import joyfill.table.Column

/**
 * Applies a change log to the document editor.
 * 
 * This function processes various types of changes (row updates, creates, moves, deletes)
 * and applies them to the appropriate table fields. It's designed to handle change logs
 * that come from external sources or event systems, allowing for programmatic updates
 * to table data.
 * 
 * The function supports all major table operations:
 * - **Row Updates**: Modifies existing row data
 * - **Row Creation**: Adds new rows to tables
 * - **Row Movement**: Reorders rows within tables
 * - **Row Deletion**: Removes rows from tables
 * 
 * **Usage Examples:**
 * ```kotlin
 * // Apply a row update change
 * val changeLog = createRowUpdateChangeLog(
 *     fieldId = "employees_table",
 *     rowId = "row_123",
 *     cellUpdates = mapOf(
 *         nameColumn to "Updated Name",
 *         ageColumn to 35.0
 *     )
 * )
 * editor.change(changeLog)
 * 
 * // Apply a row creation change
 * val changeLog = createRowCreateChangeLog(
 *     fieldId = "employees_table",
 *     targetIndex = 0,
 *     cellUpdates = mapOf(
 *         nameColumn to "New Employee",
 *         ageColumn to 28.0
 *     )
 * )
 * editor.change(changeLog)
 * 
 * // Apply a row move change
 * val changeLog = createRowMoveChangeLog(
 *     fieldId = "employees_table",
 *     rowId = "row_123",
 *     toIndex = 0
 * )
 * editor.change(changeLog)
 * 
 * // Apply a row deletion change
 * val changeLog = createRowDeleteChangeLog(
 *     fieldId = "employees_table",
 *     rowId = "row_123"
 * )
 * editor.change(changeLog)
 * ```
 * 
 * **Change Types Supported:**
 * 
 * - **[RowUpdateChange]**: Updates existing row data with new cell values
 * - **[RowCreateChange]**: Creates a new row at a specified position
 * - **[RowMoveChange]**: Moves a row to a new position within the table
 * - **[RowDeleteChange]**: Removes a row from the table
 * 
 * **Event Integration:**
 * 
 * This function is typically used in conjunction with event systems where change logs
 * are generated from user interactions or external data sources. It ensures that all
 * changes are applied consistently and trigger appropriate validation and UI updates.
 * 
 * **Error Handling:**
 * 
 * The function gracefully handles invalid change logs by:
 * - Silently ignoring changes for non-existent fields
 * - Skipping operations on non-existent rows
 * - Continuing processing even if individual operations fail
 * 
 * @param changeLog The [ChangeLog] object containing the change information. The change log
 *                  must include a valid field ID and appropriate change data based on the
 *                  operation type (update, create, move, or delete).
 * 
 * @throws IllegalArgumentException if the change log contains invalid data or references
 *                                  non-existent fields/rows.
 * @throws IllegalStateException if the document editor is in an invalid state for the
 *                               requested operation.
 * 
 * @see [RowUpdateChange] for row update operations
 * @see [RowCreateChange] for row creation operations
 * @see [RowMoveChange] for row movement operations
 * @see [RowDeleteChange] for row deletion operations
 * @see [ChangeLog] for the complete change log structure
 */
fun DocumentEditor.change(changeLogs: List<ChangeLog>) {
    changeLogs.forEach { changeLog ->
        when (val change = changeLog.change) {
            is RowUpdateChange -> {
                val cellUpdates = hashMapOf<Column, Any?>()
                val cells = (changeLog.change as RowUpdateChange).row.cells
                cells.toMap().forEach { (columnId, value) ->
                    val cell = cells.find(columnId)
                    if (cell != null) cellUpdates[cell.column] = value
                }

                update(
                    fieldId = changeLog.fieldId,
                    cellUpdates = cellUpdates,
                    rowId = change.rowId,
                    schemaId = change.schemaId,
                )
            }

            is RowCreateChange -> {
                val cellUpdates = hashMapOf<Column, Any?>()
                val cells = (changeLog.change as RowCreateChange).row.cells
                cells.toMap().forEach { (columnId, value) ->
                    val cell = cells.find(columnId)
                    if (cell != null) cellUpdates[cell.column] = value
                }

                append(
                    fieldId = changeLog.fieldId,
                    cellUpdates = cellUpdates,
                    targetIndex = change.targetRowIndex,
                    rowId = change.row.id,
                    schemaId = change.schemaId,
                    parentPath = change.parentPath,
                )
            }

            is RowMoveChange -> {
                move(
                    fieldId = changeLog.fieldId,
                    rowId = change.rowId,
                    toIndex = change.targetRowIndex,
                    schemaId = change.schemaId,
                )
            }

            is RowDeleteChange -> {
                remove(
                    fieldId = changeLog.fieldId,
                    rowId = change.rowId,
                    schemaId = change.schemaId,
                )
            }
        }
    }
}
