package joyfill.editors.document

import joyfill.Document
import joyfill.ErrorHandler
import joyfill.IdentityGenerator
import joyfill.NoOpErrorHandler
import joyfill.events.ChangeEvent
import joyfill.parseDocument
import joyfill.toDocument
import kotlinx.serialization.json.JsonObject
import wisdom.ResolutionResourceBuilder

/**
 * Creates a new [DocumentEditor] for the given [document].
 *
 * @param [document] the document to edit.
 * @param [identity] the identity generator to use while creating items in the document
 * @param [layout] the viewer to use during viewing and updating the document
 *
 * @return a new [DocumentEditor] for the given [document].
 */
fun editorOf(
    document: Document,
    validateSchema: Boolean = true,
    identity: IdentityGenerator = IdentityGenerator.default,
    layout: LayoutConfig = LayoutConfig.convenientMobile(),
    functions: (ResolutionResourceBuilder.() -> Unit)? = null,
    onError: ErrorHandler = NoOpErrorHandler,
    onChange: (DocumentEditor.(ChangeEvent) -> Unit)? = null,
): DocumentEditor = DocumentEditorImpl(
    validateSchema = validateSchema,
    document = document,
    identity = identity,
    layout = layout,
    functions = functions,
    onError = onError,
    onChange = onChange
)

/**
 * Creates a new [DocumentEditor] for the given [json].
 *
 * @param [json] the json representation of the document to edit.
 * @param [identity] the identity generator to use while creating items in the document
 * @param [layout] the viewer to use during viewing and updating the document
 *
 * @return a new [DocumentEditor] for the given [json].
 */
fun editorOf(
    json: JsonObject,
    validateSchema: Boolean = true,
    identity: IdentityGenerator = IdentityGenerator.default,
    layout: LayoutConfig = LayoutConfig.convenientMobile(),
    functions: (ResolutionResourceBuilder.() -> Unit)? = null,
    onError: ErrorHandler = NoOpErrorHandler,
    onChange: (DocumentEditor.(ChangeEvent) -> Unit)? = null,
): DocumentEditor= editorOf(
    document = json.toDocument(),
    validateSchema = validateSchema,
    identity = identity,
    layout = layout,
    functions = functions,
    onError = onError,
    onChange = onChange
)


/**
 * Creates a new [DocumentEditor] from the given [map].
 *
 * @param [map] the document to edit.
 * @param [identity] the identity generator to use while creating items in the document
 * @param [layout] the viewer to use during viewing and updating the document
 *
 * @return a new [DocumentEditor] for the given [map].
 */
fun editorOf(
    map: MutableMap<String, Any?>,
    validateSchema: Boolean = true,
    identity: IdentityGenerator = IdentityGenerator.default,
    layout: LayoutConfig = LayoutConfig.convenientMobile(),
    functions: (ResolutionResourceBuilder.() -> Unit)? = null,
    onError: ErrorHandler = NoOpErrorHandler,
    onChange: (DocumentEditor.(ChangeEvent) -> Unit)? = null,
): DocumentEditor = editorOf(
    document = map.toDocument(),
    validateSchema = validateSchema,
    identity = identity,
    layout = layout,
    functions = functions,
    onError = onError,
    onChange = onChange
)

/**
 * Creates a new [DocumentEditor] from the given [json].
 *
 * @param [json] the document to edit.
 * @param [identity] the identity generator to use while creating items in the document
 * @param [layout] the viewer to use during viewing and updating the document
 *
 * @return a new [DocumentEditor] for the given [json].
 */
fun editorOf(
    json: String,
    validateSchema: Boolean = true,
    identity: IdentityGenerator = IdentityGenerator.default,
    layout: LayoutConfig = LayoutConfig.convenientMobile(),
    functions: (ResolutionResourceBuilder.() -> Unit)? = null,
    onError: ErrorHandler = NoOpErrorHandler,
    onChange: (DocumentEditor.(ChangeEvent) -> Unit)? = null,
): DocumentEditor = editorOf(
    document = parseDocument(json),
    validateSchema = validateSchema,
    identity = identity,
    layout = layout,
    functions = functions,
    onError = onError,
    onChange = onChange
)