package joyfill.editors.date.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.Field
import joyfill.collections.PageCollection
import joyfill.components.Component
import joyfill.date.DateComponent
import joyfill.date.DateField
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditor
import joyfill.editors.date.DateEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.notify
import joyfill.editors.utils.localTimeZone
import joyfill.editors.utils.timezoneFromId
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.tools.validation.ComponentValidity
import joyfill.tools.visibility.Visibility
import joyfill.utils.TIME_ZONE
import kotlinx.datetime.TimeZone
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class DateEditorImpl(
    override val component: DateComponent,
    override val format: String?,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library?,
    document: Document,
    pages: PageCollection?,
    onChange: ((ChangeEvent) -> Unit)?,
    onValidate: (ComponentValidity) -> Unit,
    identifier: String,
    fieldId: String = component.id,
    dispatcher: EventDispatcher?,
) : DateEditor, AbstractComponentEditor(
    document = document,
    pages = pages,
    component = component,
    onValidate = onValidate,
    onChange = onChange,
    identifier = identifier,
    parentFieldId = fieldId,
    dispatcher = dispatcher,
    dependents = dependents,
    resolver = resolver,
    library = library
) {

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        hidden = value
        state.value = state.value.copy(
            visibility = if (value) Visibility.Hidden else Visibility.Visible,
        )
    }

    override val timeZone: TimeZone
        get() {
            val field = component as? DateField
            return timezoneFromId(field?.timeZone) ?: localTimeZone
        }

    override val state by lazy {
        val field = component as? Field

        val initialValue = if (field != null && field.formulas.isNotEmpty()) {
            val result = library?.call(field.id, resolver) as? Long
            result ?: component.value
        } else {
            component.value
        }

        value(initialValue)
        mutableLiveOf(
            DateEditorStateImpl(
                data = component.value,
                visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
                validity = validate()
            )
        )
    }

    override val isValid: Boolean
        get() {
            val value = component.value
            return value != null && value != 0L
        }

    override fun value(millis: Long?, origin: Origin) {
        if (component.value == millis) return

        val field = component as? DateField
        if (field != null && field.timeZone.isNullOrEmpty()) {
            field.timeZone = localTimeZone.id
        }
        component.value = millis
        resolveConditions()
        state.value = state.value.copy(
            data = component.value,
            validity = validate(),
            visibility = if (component.hidden) Visibility.Hidden else Visibility.Visible,
        )
        updateDependentsValues()
        if (origin.notify) notifyChange(
            mutableMapOf<String, Any?>(
                Component::value.name to millis,
                TIME_ZONE to timeZone.id,
            )
        )
    }
}