package joyfill.editors.components.internal

import joyfill.Document
import joyfill.Field
import joyfill.collections.PageCollection
import joyfill.components.Component
import joyfill.conditions.field.FieldConditionDefinition
import joyfill.editors.chart.ChartEditor
import joyfill.editors.chart.internal.ChartEditorImpl
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.components.AbstractCompStringEditor
import joyfill.editors.components.ComponentEditor
import joyfill.editors.date.DateEditor
import joyfill.editors.dropdown.DropdownEditor
import joyfill.editors.event.EventTrigger
import joyfill.editors.multi_select.MultiSelectEditor
import joyfill.editors.number.NumberEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.TableEditor
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.schemas.Schema
import joyfill.table.Cell
import joyfill.tools.validation.ComponentInvalid
import joyfill.tools.validation.ComponentValid
import joyfill.tools.validation.ComponentValidity
import joyfill.tooltip.ToolTip
import wisdom.ResolutionResource
import wisdom.ast.Library

internal abstract class AbstractComponentEditor(
    private val document: Document,
    private val pages: PageCollection?,
    override val component: Component,
    val onValidate: (ComponentValidity) -> Unit = {},
    onChange: ((ChangeEvent) -> Unit)?,
    identifier: String,
    parentFieldId: String = component.id,
    private val dispatcher: EventDispatcher? = null,
    private val dependents: () -> List<ComponentEditor>,
    private val resolver: ResolutionResource,
    private val library: Library?,
) : ComponentEditor,
    Schema by component,
    EventTrigger<Component>(
        document = document,
        component = component,
        fieldId = parentFieldId,
        fieldIdentifier = identifier,
        onChange = onChange
    ) {

    override fun show() = setHiddenTo(false, HiddenReason.MANUAL, true)

    override fun hide() = setHiddenTo(true, HiddenReason.MANUAL, true)

    private val conditionEvaluator by lazy { FieldConditionEvaluator(::getField) }

    private fun getField(key: String): Field? =
        document.fields.find {
            it.identifier == key || it.id == key || it.title == key
        }

    private fun FieldConditionDefinition.isInvolved(with: Field): Boolean {
        return with.identifier == field || with.id == field || with.title == field
    }

    override fun resolveConditions() {
        if (dispatcher == null || component is Cell) return

        // resolving conditions for fields
        document.fields.filter { it.logic != null }.forEach { field ->
            val logic = field.logic ?: return@forEach
            if (logic.conditions.none { it.isInvolved(with = component as Field) }) return@forEach
            val value = conditionEvaluator.evaluateConditions(
                logic = logic,
                currentHidden = field.hidden,
                initialHidden = field.initialHiddenState,
            )

            dispatcher.setHidden(
                component = field.id,
                reason = HiddenReason.CONDITIONAL_LOGIC,
                value = value,
            )
        }

        // resolving conditions for pages
        pages?.raw()?.filter { it.logic != null }?.forEach { p ->
            val logic = p.logic ?: return@forEach
            if (logic.conditions.none { it.isInvolved(with = component as Field) }) return@forEach

            pages.setHidden(
                key = p.id,
                value =
                    conditionEvaluator.evaluateConditions(
                        logic = logic,
                        currentHidden = p.hidden,
                        initialHidden = p.initialHiddenState,
                    )
            )
        }
    }

    override val tip: ToolTip?
        get() =
            when (val c = component) {
                is Field -> c.tip
                else -> null
            }


    override var identifier: String
        get() =
            when (val c = component) {
                is Field -> c.identifier
                else -> c.id
            }
        set(value) {
            when (val c = component) {
                is Field -> c.identifier = value
                else -> c.id = value
            }
        }

    override val disabled: Boolean
        get() =
            when (val c = component) {
                is Field -> c.disabled
                else -> false
            }

    open val isValid: Boolean = false

    override fun updateDependentsValues() {
        // Skip updating dependents if this field has no library (indicating it's circular)
        // This prevents the iterative resolution that leads to stabilized values
        if (library == null) return

        for (dependent in dependents()) {
            if (!library.routines.containsKey(dependent.id)) {
                continue
            }

            when (dependent) {
                is NumberEditor -> {
                    val result = library.call(dependent.id, resolver)
                    val doubleResult = when (result) {
                        is Number -> result.toDouble()
                        else -> result as? Double
                    } ?: 0.0
                    if (dependent.state.value.data == doubleResult) continue
                    dependent.value(doubleResult)
                }

                is DropdownEditor -> {
                    val result = library.call(dependent.id, resolver) as? String
                    if (dependent.state.value.data?.value == result) continue
                    dependent.value(result)
                }

                is MultiSelectEditor -> {
                    val result = library.call(dependent.id, resolver) as? List<String>
                    if (result == null || dependent.state.value.data.map { it.id } == result) continue
                    val options = dependent.options.filter { it.id in result || it.value in result }
                    dependent.value(options)
                }

                is DateEditor -> {
                    val result = library.call(dependent.id, resolver)
                    val longResult = when (result) {
                        is Number -> result.toLong()
                        else -> result as? Long
                    }
                    if (dependent.state.value.data == longResult) continue
                    dependent.value(longResult)
                }

                is AbstractCompStringEditor -> {
                    val result = library.call(dependent.id, resolver)
                    val stringResult = when (result) {
                        is String -> result
                        is Number -> result.toString()
                        is Boolean -> result.toString()
                        null -> null
                        else -> result.toString()
                    } ?: ""
                    if (dependent.state.value.data == stringResult) continue
                    dependent.value(stringResult)
                }

                // Add support for complex field types that can be formula targets
                is CollectionEditor -> {
                    val result = library.call(dependent.id, resolver)
                    // Collection editors don't have a direct value setter, but they should re-evaluate their formulas
                    // We trigger the update by calling updateDependentsValues on the collection editor itself
                    dependent.updateDependentsValues()
                }

                is ChartEditor -> {
                    val result = library.call(dependent.id, resolver)
                    if (dependent is ChartEditorImpl) {
                        val convertedResult = dependent.linesFrom(result)
                        if (convertedResult != null) {
                            dependent.value(convertedResult)
                        }
                    }
                }

                is TableEditor -> {
                    val result = library.call(dependent.id, resolver)
                    // Table editors should re-evaluate their formulas
                    dependent.updateDependentsValues()
                }
            }
        }
    }

    override fun validate(): ComponentValidity {
        val validity = if (!component.required || isValid) {
            ComponentValid.empty(component)
        } else {
            ComponentInvalid(component, listOf("Component ${component.title} is required"))
        }
        onValidate(validity)
        return validity
    }
}