package joyfill.editors.collection.internal

import cinematic.mutableLiveOf
import joyfill.IdentityGenerator
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.TableFilter
import joyfill.editors.table.ColumnFilter
import joyfill.editors.table.ColumnSorter
import joyfill.editors.table.ColumnTransformer
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.SortingOption
import joyfill.editors.table.TableEditor
import joyfill.editors.table.internal.ColumnSorterImpl
import joyfill.editors.table.internal.TableFilterStateImpl
import joyfill.editors.table.internal.getRootRowId
import joyfill.table.Column
import joyfill.table.TableSchema
import joyfill.utils.Option

internal data class TableFilterImpl(
    override val schema: TableSchema,
    private val root: TableEditor,
    private val getEditors: () -> List<TableEditor>,
    private val identity: IdentityGenerator,
    private val afterFiltering: () -> Unit = {},
    private val beforeFiltering: () -> Unit = {},
    private val disableFilter: () -> Unit = {},
    private val host: CollectionEditor? = null,
) : TableFilter {

    override val state by lazy {
        mutableLiveOf(TableFilterStateImpl.empty(columns = schema.columns.filterNot { it.hidden }))
    }

    override fun addFilter(): ColumnFilter {
        val filter = ColumnFilter.from(null, identity = identity)
        state.value = state.value.copy(filters = state.value.filters + filter)
        return filter
    }

    override fun addSorter(): ColumnSorter? {
        if (state.value.sorter != null) return null
        val sorter = ColumnSorter.from(null, identityGenerator = identity)
        state.value = state.value.copy(sorter = sorter)
        return sorter
    }

    override fun ColumnTransformer.select(column: Column?): ColumnTransformer? {
        if (column == null) return null

        val isSorter = this is ColumnSorter

        val transformer = if (isSorter) {
            ColumnSorter.from(column, identityGenerator = identity)
        } else {
            ColumnFilter.from(column, identity = identity)
        }

        if (isSorter) {
            state.value = state.value.copy(sorter = transformer as ColumnSorter)
        } else {
            state.value = state.value.copy(
                filters = state.value.filters.map { if (it.id == id) transformer as ColumnFilter else it },
                filterColumns = state.value.filterColumns - column,
            )
        }

        return transformer
    }

    override fun removeSorter() {
        val currentValue = state.value
        val hasFilters = currentValue.filters.isNotEmpty()
        // PS(Mohit): We do not check for sorter being present since we removed the sorter
        if (hasFilters) {
            state.value = currentValue.copy(sorter = null)
        } else {
            disable()
        }
    }

    override fun removeFilter(id: String) {
        val currentValue = state.value
        val removedFilter = currentValue.filters.firstOrNull {
            it.id == id
        }

        val updatedFilters = currentValue.filters.filterNot { it.id == id }
        val updatedColumns= currentValue.columns.filter { it !in updatedFilters.map { filter -> filter.column } }

        val hasFilters = updatedFilters.isNotEmpty()
        if (hasFilters || currentValue.sorter != null) {
            state.value = currentValue.copy(
                filters = updatedFilters,
                filterColumns = updatedColumns,
            )
        } else {
            disable()
        }
    }

    override fun apply() {
        if (state.value.isActive) {
            disablePreviousFilters()
            beforeFiltering()
            val editors = getEditors()
            val allFilteredRows = editors.flatMap {
                it.rows.applyFilters(it)
            }

            host?.let {
                if (host is CollectionEditorImpl) {
                    host.rootsToShow = allFilteredRows.mapTo(mutableSetOf()) { getRootRowId(it.id) }
                }
            }
            afterFiltering()
        }
    }

    override fun disable() {
        state.value = state.value.copy(
            filters = emptyList(),
            sorter = null,
            filterColumns = state.value.columns,
        )

        if (state.value.isActive) {
            disablePreviousFilters()
        }
        disableFilter()
        state.value = state.value.copy(isActive = false)
    }

    private fun disablePreviousFilters() {
        root.setHiddenTo(false, HiddenReason.FILTER, false)
        getEditors().forEach { it.rows.disableFilters() }
    }

    override fun enable() {
        val current = state.value
        val filters = current.filters.filter { it.column != null }
        val sorter = if (current.sorter?.column == null) null else current.sorter
        state.value = current.copy(
            filters = filters,
            sorter = sorter,
            isActive = filters.isNotEmpty() || sorter != null,
        )
    }

    override fun ColumnFilter.updateFilterValue(newValue: Any?): ColumnFilter {
        state.value = state.value.copy(
            filters = state.value.filters.map { transformer ->
                if (transformer.id == id) {
                    when (transformer) {
                        is ColumnFilter.EmptyColumnFilter -> transformer.copy(value = newValue as? String)
                        is ColumnFilter.TextColumnFilter -> transformer.copy(value = newValue as? String)
                        is ColumnFilter.NumberColumnFilter -> transformer.copy(value = newValue as? String)
                        is ColumnFilter.BarcodeColumnFilter -> transformer.copy(value = newValue as? String)
                        is ColumnFilter.DropdownColumnFilter -> transformer.copy(value = newValue as? Option)
                        is ColumnFilter.MultiselectColumnFilter -> transformer.copy(value = newValue as? Option)
                    }
                } else {
                    transformer
                }
            }
        )
        return state.value.filters.first { it.id == id }
    }

    override fun updateSorterValue(sort: SortingOption): ColumnSorter? {
        state.value = state.value.copy(
            sorter = state.value.sorter?.let { transformer ->
                ColumnSorterImpl(
                    column = transformer.column,
                    type = transformer.type,
                    id = transformer.id,
                    sort = sort
                )
            }
        )
        return state.value.sorter
    }
}