package joyfill.editors.collection.internal

import cinematic.mutableLiveOf
import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collections.PageCollection
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.RowManager
import joyfill.editors.collection.Tables
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.internal.AbstractRowManager
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.table.Column
import joyfill.table.Row
import joyfill.table.TableSchema
import joyfill.tools.validation.Valid
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class RowManagerImpl(
    override val id: String,
    document: Document,
    pages: PageCollection,
    private val host: CollectionEditor,
    private val table: TableSchema,
    private val initialHiddenMap: Map<String, Boolean>,
    override val row: Row,
    override val parent: RowManager?,
    columns: List<Column>,
    identity: IdentityGenerator,
    onRowChange: ((columnId: String, ChangeEvent) -> Unit)?,
    onChange: ((ChangeEvent) -> Unit)?,
    override val next: () -> RowManager?,
    override val prev: () -> RowManager?,
    private val setRowHidden: (String) -> Unit,
    private val setTableHidden: (String) -> Unit,
    private val onAppend: (tableId: String, rowManager: RowManager, index: Int) -> Unit,
    private val resolve: (id: String, type: ResolveType) -> Unit,
    override val tableSchemaId: String = table.id,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library? = null,
    dispatcher: EventDispatcher? = null,
) : AbstractRowManager(
    row = row,
    document = document,
    columns = columns,
    required = true,
    identity = identity,
    onChange = onRowChange,
    fieldId = host.id,
    identifier = host.identifier,
    dependents = dependents,
    resolver = resolver,
    library = library,
    dispatcher = dispatcher
), RowManager {
    override val tables: Tables by  lazy {
        TablesImpl(
            id = id,
            host = host,
            identity = identity,
            initialHiddenMap = initialHiddenMap,
            parent = this,
            table = table,
            next = next,
            prev = prev,
            document = document,
            pages = pages,
            onAppend = onAppend,
            onChange = onChange,
            setRowHidden = setRowHidden,
            setTableHidden = setTableHidden,
            resolve = resolve,
            dependents = dependents,
            resolver = resolver,
            library = library,
            dispatcher = dispatcher,
        )
    }

    override fun setHidden(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        super.setHidden(value, reason, liveUpdate)
        if (liveUpdate)
            setRowHidden(id)
    }

    override val isValid = mutableLiveOf(true)

    private val hasRequiredNestedValidation by lazy {
        hasAnyRequiredNestedFields()
    }

    private fun RowManager.deepValidate(): Boolean {
        val allTables = tables.all()
        if (allTables.isEmpty()) return true

        for (table in allTables) {
            if (table.state.value.visibility.isHidden()) continue

            val tableIsValid = table.validate() is Valid
            if (!tableIsValid) return false

            val displayingRows = table.rows.state.value.displaying()
            for (row in displayingRows) {
                if (row.state.value.visibility.isHidden()) continue
                if (!row.validateRow()) return false
                if (!row.deepValidate()) return false
            }
        }
        return true
    }

    override fun validate(): Boolean {
        if (state.value.visibility.isHidden()) return true

        val newValid = if (!hasRequiredNestedValidation) {
            state.value.allCellsValid
        } else {
            state.value.allCellsValid && deepValidate()
        }
        
        if (isValid.value != newValid || isValid.history.isEmpty()) {
            this@RowManagerImpl.isValid.value = newValid
            resolve(this@RowManagerImpl.id, ResolveType.Validation)
            parent?.validate()
        }
        
        return newValid
    }

    private fun hasAnyRequiredNestedFields(): Boolean {
        val allNestedSchemas = getAllNestedTableSchemas()

        return allNestedSchemas.any { schema ->
            schema.required || schema.columns.any { it.required }
        }
    }

    private fun getAllNestedTableSchemas(): List<TableSchema> {
        val allSchemas = mutableListOf<TableSchema>()
        val visitedSchemas = mutableSetOf<String>()

        tables.all().forEach { table ->
            collectNestedSchemas(table.component, allSchemas, visitedSchemas)
        }

        return allSchemas
    }

    private fun collectNestedSchemas(
        schema: TableSchema,
        collector: MutableList<TableSchema>,
        visited: MutableSet<String>
    ) {
        if (schema.id in visited) return
        visited.add(schema.id)

        collector.add(schema)

        schema.children.forEach { childId ->
            val childSchema = host.component.schema.all().find { it.id == childId || it.title == childId }
            if (childSchema != null) {
                collectNestedSchemas(childSchema, collector, visited)
            }
        }
    }
}
