package joyfill.editors.collection.internal

import joyfill.Document
import joyfill.IdentityGenerator
import joyfill.collections.PageCollection
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.collection.RowManager
import joyfill.editors.collection.TableEditor
import joyfill.editors.collection.TableFilter
import joyfill.editors.collection.entries.RowEntry
import joyfill.editors.components.ComponentEditor
import joyfill.editors.table.Columns
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.Origin
import joyfill.editors.table.RowResult
import joyfill.editors.table.RowVisibility
import joyfill.editors.table.internal.AbstractTableEditor
import joyfill.editors.table.internal.ColumnsImpl
import joyfill.editors.table.internal.TableConditionEvaluator
import joyfill.editors.table.notify
import joyfill.editors.utils.localTimeZone
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import joyfill.events.RowChange
import joyfill.events.Target
import joyfill.table.Cell
import joyfill.table.Row
import joyfill.table.TableComponent
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class NestedTableEditor(
    override var id: String,
    private val host: CollectionEditor,
    override val component: TableComponent,
    override val parent: RowManager? = null,
    private val identity: IdentityGenerator,
    private val initialHiddenMap: Map<String, Boolean>,
    private val next: () -> RowManager?,
    private val prev: () -> RowManager?,
    private val setRowHidden: (String) -> Unit,
    private val setTableHidden: (String) -> Unit,
    private val onAppend: (tableId: String, rowManager: RowManager, index: Int) -> Unit,
    private val resolve: (id: String, type: ResolveType) -> Unit,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    dependents: () -> List<ComponentEditor>,
    resolver: ResolutionResource,
    library: Library? = null,
    dispatcher: EventDispatcher? = null,
) : AbstractTableEditor(
    component = component,
    initialHidden = initialHiddenMap[component.id] ?: false,
    document = document,
    pages = pages,
    onChange = onChange,
    identifier = host.identifier,
    fieldId = host.id,
    dispatcher = dispatcher,
    dependents = dependents,
    resolver = resolver,
    library = library
), TableEditor {

    private val conditionEvaluator by lazy { TableConditionEvaluator(::getCell) }

    override fun getCell(column: String): Cell? = parent?.row?.cells?.find(column)

    override val columns: Columns by lazy { ColumnsImpl(component.columns) }

    override val filters: TableFilter by lazy {
        TableFilterImpl(
            schema = component,
            getEditors = { listOf(this) },
            identity = identity,
            root = host.root(),
        )
    }

    override fun setHiddenTo(value: Boolean, reason: HiddenReason, liveUpdate: Boolean) {
        when (reason) {
            HiddenReason.CONDITIONAL_LOGIC -> setHiddenToFromConditions(value, liveUpdate)
            HiddenReason.FILTER -> setHiddenToFromFilters(value, liveUpdate)
            HiddenReason.MANUAL -> setHiddenToFromManual(value)
        }
    }

    private fun setHiddenToFromManual(value: Boolean) {
        hidden = value

        val newVisibility = if (value) RowVisibility.Hidden(HiddenReason.MANUAL) else RowVisibility.Visible
        state.value = state.value.copy(visibility = newVisibility)

        setTableHidden(id)

        rows.state.value.all.forEach { rowManager ->
            rowManager.setHidden(value, HiddenReason.MANUAL, true)
            rowManager.tables.all().forEach { tableEditor ->
                tableEditor.setHiddenTo(value, HiddenReason.MANUAL, true)
            }
        }
    }

    private fun setHiddenToFromConditions(value: Boolean, liveUpdate: Boolean) {
        val currentVisibility = state.value.visibility
        val currentReason = currentVisibility.reason()
        if (currentReason == HiddenReason.FILTER) return

        hidden = value

        val newVisibility = if (value) RowVisibility.Hidden(HiddenReason.CONDITIONAL_LOGIC) else RowVisibility.Visible
        state.value = state.value.copy(visibility = newVisibility)

        if (liveUpdate) setTableHidden(id)

        for (rowManager in rows.state.value.all) {
            val reason = rowManager.state.value.visibility.reason()
            if (reason == HiddenReason.FILTER) continue

            if (value) {
                rowManager.setHidden(true, HiddenReason.CONDITIONAL_LOGIC, liveUpdate)
                rowManager.tables.all().forEach { tableEditor ->
                    tableEditor.setHiddenTo(true, HiddenReason.CONDITIONAL_LOGIC, liveUpdate)
                }
            } else {
                rowManager.setHidden(false, HiddenReason.CONDITIONAL_LOGIC, liveUpdate)
                rowManager.tables.all().forEach { tableEditor ->
                    if (tableEditor !is NestedTableEditor) return@forEach
                    val hidden = tableEditor.shouldHideSchema() ?: false
                    tableEditor.setHiddenTo(hidden, HiddenReason.CONDITIONAL_LOGIC, liveUpdate)
                }
            }
        }
    }

    private fun setHiddenToFromFilters(value: Boolean, liveUpdate: Boolean) {
        val currentVisibility = state.value.visibility
        val currentReason = currentVisibility.reason()
        if (currentReason == HiddenReason.CONDITIONAL_LOGIC) return

        val newVisibility = if (value) RowVisibility.Hidden(HiddenReason.FILTER) else RowVisibility.Visible
        state.value = state.value.copy(visibility = newVisibility)

        if (liveUpdate) setTableHidden(id)

        for (rowManager in rows.state.value.all) {
            if (rowManager.state.value.visibility.reason() == HiddenReason.CONDITIONAL_LOGIC) continue

            if (value) {
                rowManager.setHidden(true, HiddenReason.FILTER, liveUpdate)
                rowManager.tables.all().forEach { tableEditor ->
                    tableEditor.setHiddenTo(true, HiddenReason.FILTER, liveUpdate)
                }
            } else {
                rowManager.setHidden(false, HiddenReason.FILTER, liveUpdate)
                rowManager.tables.all().forEach { tableEditor ->
                    if (tableEditor !is NestedTableEditor) return@forEach
                    val hidden = tableEditor.shouldHideSchema() ?: false
                    tableEditor.setHiddenTo(hidden, HiddenReason.FILTER, liveUpdate)
                }
            }
        }
    }

    override val rows by lazy {
        NestedRowsImpl(
            host = host,
            document = document,
            pages = pages,
            table = this,
            identity = identity,
            next = next,
            parent = parent,
            setRowHidden = setRowHidden,
            setTableHidden = setTableHidden,
            initialHiddenMap = initialHiddenMap,
            prev = prev,
            onAppend = onAppend,
            onCreate = ::onCreate,
            onDelete = ::onDelete,
            onMove = ::onMove,
            onRowChange = ::onRowChange,
            onChange = onChange,
            resolve = resolve,
            dependents = dependents,
            resolver = resolver,
            library = library,
        )
    }

    init {
        resolveConditions()
    }

    override fun resolveConditions() {
        val value = shouldHideSchema() ?: return
        val currentReason = state.value.visibility.reason()
        setHiddenTo(value, currentReason ?: HiddenReason.CONDITIONAL_LOGIC, true)
    }

    private fun onCreate(row: RowManager, index: Int, origin: Origin) {
        val updatedRows = state.value.rows.toMutableList()
        updatedRows.add(index, row)

        //we can remove these since the ui state does not depend on this
        state.value = state.value.copy(rows = updatedRows)
        val visibility = state.value.visibility
        val isHidden = visibility.isHidden()

        if (isHidden) {
            val hiddenReason = visibility.reason() ?: HiddenReason.CONDITIONAL_LOGIC
            row.setHidden(isHidden, hiddenReason, false)
            row.tables.all().forEach { tableEditor ->
                tableEditor.setHiddenTo(isHidden, hiddenReason, false)
            }
        } else {
            row.tables.all().forEach {
                it.resolveConditions()
            }
        }

        state.value = state.value.copy(validity = validate(updatedRows))
        row.validate()
        onAppend(id, row, index)

        if (origin.notify) {
            val base = buildRowCreateChangeValue(rowManager = row, index = index, includeChildren = true)
            base[RowChange::parentPath.name] = parentPath
            base[RowChange::schemaId.name] = component.id

            notifyChange(base, Target.field_value_rowCreate)
        }
    }

    private fun onRowChange(columnId: String, changeEvent: ChangeEvent, row: Row) {
        if (row.timeZone.isNullOrEmpty() && row.cells.date(columnId) != null) row.timeZone = localTimeZone.id
        changeEvent.changelogs.forEach { changeLog ->
            val base = buildRowUpdateChangeValue(columnId, row, changeLog)
            base[RowChange::parentPath.name] = parentPath
            base[RowChange::schemaId.name] = component.id

            notifyChange(base, Target.field_value_rowUpdate)
        }

        val rowManager = rows.find(row.id) ?: return
        val tables = rowManager.tables.all()

        if (tables.isEmpty() || tables.all { it.component.tableLogic == null }) {
            rowManager.validate()
            return
        }
        tables.forEach { it.resolveConditions() }
        rowManager.validate()
    }

    private fun onDelete(rows: List<RowManager>, origin: Origin) {
        if (origin.notify) {
            rows.forEach {
                val base = buildRowDeleteChangeValue(it)
                base[RowChange::parentPath.name] = parentPath
                base[RowChange::schemaId.name] = component.id

                notifyChange(base, Target.field_value_rowDelete)
            }
        }
        //we can remove these since the ui state does not depend on this
        val newRows = state.value.rows - rows.toSet()
        state.value = state.value.copy(
            rows = newRows,
            validity = validate(newRows)
        )
    }

    private fun onMove(rowResult: RowResult<Row>, origin: Origin) {
        if (rowResult.index < 0) return
        if (origin.notify) {
            val base = buildRowMoveChangeValue(rowResult.row, rowResult.index)
            base[RowChange::parentPath.name] = parentPath
            base[RowChange::schemaId.name] = component.id

            notifyChange(base, Target.field_value_rowMove)
        }
        //we can remove these since the ui state does not depend on this
        val updatedRows = component.rowOrder.mapNotNull { id ->
            state.value.rows.find { it.row.id == id }
        }
        state.value = state.value.copy(rows = updatedRows)
    }

    private fun shouldHideSchema(): Boolean? {
        val logic = component.tableLogic ?: return null
        val initialHiddenValue = initialHiddenMap[component.id] ?: return null
        return conditionEvaluator.evaluateConditions(
            logic = logic,
            currentHidden = state.value.visibility.isHidden(),
            initialHidden = initialHiddenValue
        )
    }

    private val parentPath: String by lazy {
        val parentManager = parent ?: return@lazy ""

        val rowEntries = host.state.value.entries.filterIsInstance<RowEntry>()
        val parentRowEntry = rowEntries.firstOrNull { it.manager.row.id == parentManager.row.id }
        if (parentRowEntry == null) return@lazy ""


        val parentTableEditor = parentRowEntry.table
        val parentIndex = parentRowEntry.number - 1
        val currentSchemaId = component.identifier

        val isNestedTable = parentTableEditor.parent != null && parentTableEditor is NestedTableEditor
        if (!isNestedTable) return@lazy "$parentIndex.$currentSchemaId"

        val ancestorPath = parentTableEditor.parentPath
        if (ancestorPath.isNotEmpty()) {
            "$ancestorPath.$parentIndex.$currentSchemaId"
        } else {
            "$parentIndex.$currentSchemaId"
        }
    }
}
