package joyfill.editors.collection.internal

import joyfill.editors.collection.RowManager
import joyfill.editors.collection.entries.LazyEntry
import joyfill.editors.collection.entries.RowEntry
import joyfill.editors.collection.entries.TableEntry
import joyfill.editors.table.RowResult
import joyfill.table.Row

internal fun CollectionEditorImpl.expand(tableEntry: TableEntry) {
    val currentEntries = state.value.entries
    val expandedRowIds = currentEntries
        .filterIsInstance<RowEntry>()
        .filter { it.expanded }
        .map { it.manager.row.id }
        .toSet()

    val expandedEntries = tableEntry.expanded(expandedRowIds, currentEntries)
    val entries = state.value.entries.toMutableList()
    val index = entries.indexOf(tableEntry).takeIf { it >= 0 } ?: entries.size
    
    entries.removeAll { it.id.contains(tableEntry.id) }
    entries.addAll(index, expandedEntries)
    
    state.value = state.value.copy(entries = entries)
    rebuildEntryIndexes()
}

internal fun CollectionEditorImpl.moveRowToIndex(tableEntry: TableEntry, rowResult: RowResult<Row>) {
    if (rowResult.index < 0) return
    
    val entries = state.value.entries.toMutableList()
    val tableIndex = entries.indexOf(tableEntry).takeIf { it >= 0 } ?: return
    val movedRowId = rowResult.row?.id ?: return
    
    // Find the row entry that was moved
    val movedRowIndex = entries.indexOfFirst { entry ->
        entry is RowEntry && 
        entry.manager.row.id == movedRowId && 
        entry.table.id == tableEntry.table.id
    }.takeIf { it >= 0 } ?: return
    
    // Extract row with all nested content
    val contentEndIndex = findRowContentEnd(entries, movedRowIndex)
    val rowWithContent = entries.subList(movedRowIndex, contentEndIndex + 1).toList()
    
    // Remove from current position
    for (i in contentEndIndex downTo movedRowIndex) {
        entries.removeAt(i)
    }
    
    // Calculate new insertion position
    val insertionIndex = calculateRowInsertionIndex(entries, tableEntry, rowResult.index, tableIndex)
    
    // Insert at new position
    entries.addAll(insertionIndex, rowWithContent)
    updateRowNumbers(entries, tableEntry.table.id)
    
    state.value = state.value.copy(entries = entries)
    rebuildEntryIndexes()
}

internal fun CollectionEditorImpl.removeRows(deletedManagers: List<RowManager>): MutableList<LazyEntry> {
    val entries = state.value.entries.toMutableList()
    
    // Build manager ID to index mapping in single pass - fix the casting issue
    val managerToIndexMap = mutableMapOf<String, Int>()
    entries.forEachIndexed { index, entry ->
        if (entry is RowEntry) {
            managerToIndexMap[entry.manager.id] = index
        }
    }
    
    // Calculate deletion ranges for all managers
    val deletionRanges = deletedManagers.mapNotNull { manager ->
        val startIndex = managerToIndexMap[manager.id] ?: return@mapNotNull null
        val endIndex = findRowContentEnd(entries, startIndex)
        startIndex..endIndex
    }.sortedByDescending { it.first } // Process in reverse order
    
    // Remove entries for each range
    deletionRanges.forEach { range ->
        for (i in range.last downTo range.first) {
            if (i < entries.size) {
                entries.removeAt(i)
            }
        }
    }
    
    return entries
}

/**
 * Calculate insertion index for row movement using shared logic.
 */
private fun calculateRowInsertionIndex(
    entries: List<LazyEntry>,
    tableEntry: TableEntry,
    targetIndex: Int,
    tableIndex: Int
): Int {
    val tableRows = findTableRows(entries, tableEntry, tableIndex)
    
    return when {
        tableRows.isEmpty() || targetIndex == 0 -> tableIndex + 1
        targetIndex >= tableRows.size -> {
            val lastRowIndex = tableRows.lastOrNull()?.second ?: return tableIndex + 1
            findRowContentEnd(entries, lastRowIndex) + 1
        }
        else -> tableRows[targetIndex].second
    }
}