package joyfill.editors.collection.internal

import joyfill.editors.collection.RowManager
import joyfill.editors.collection.entries.LazyEntry
import joyfill.editors.collection.entries.RowEntry
import joyfill.editors.collection.entries.TableEntry
import joyfill.tools.validation.Valid

internal fun CollectionEditorImpl.expand(rowEntry: RowEntry) {
    if (rowEntry.expanded) return
    
    val nestedTables = rowEntry.manager.tables.all()
    val tableEntriesToAdd = nestedTables.map { table ->
        TableEntry(
            level = rowEntry.level + 1,
            table = table,
            expanded = true,
            hidden = table.state.value.visibility.isHidden(),
            parent = rowEntry.manager.row,
            isValid = table.state.value.validity is Valid
        )
    }

    updateEntries(rowEntry, tableEntriesToAdd)
}

internal fun CollectionEditorImpl.insertRowAtPosition(tableId: String, rowManager: RowManager, position: Int) {
    val entries = state.value.entries.toMutableList()
    
    // Find both the table entry and its index in a single pass
    val (tableEntry, tableIndex) = entries.withIndex()
        .firstOrNull { (_, entry) -> entry.id == tableId && entry is TableEntry }
        ?.let { (index, entry) -> entry as TableEntry to index }
        ?: return

    val newRowEntry = createRowEntry(rowManager, tableEntry)

    val insertionIndex = calculateInsertionIndex(entries, tableEntry, tableIndex, position)

    //we are guaranteed to have at least one row now, so its always valid
    val updatedTableEntry = tableEntry.copy(
        isValid = true
    )
    
    entries[tableIndex] = updatedTableEntry
    
    entries.add(insertionIndex, newRowEntry)
    updateRowNumbers(entries, tableEntry.table.id)
    
    state.value = state.value.copy(entries = entries)
    rebuildEntryIndexes()
}

private fun createRowEntry(rowManager: RowManager, tableEntry: TableEntry): RowEntry {
    return RowEntry(
        level = tableEntry.level,
        manager = rowManager,
        expanded = false,
        hidden = rowManager.state.value.visibility.isHidden(),
        isValid = rowManager.isValid.value,
        table = tableEntry.table,
        number = 0,
    )
}

private fun calculateInsertionIndex(
    entries: List<LazyEntry>,
    tableEntry: TableEntry,
    tableIndex: Int,
    position: Int
): Int {
    val tableRows = findTableRows(entries, tableEntry, tableIndex)
    
    return when {
        tableRows.isEmpty() || position == 0 -> tableIndex + 1
        position >= tableRows.size -> {
            val lastRowIndex = tableRows.lastOrNull()?.second ?: return tableIndex + 1
            findRowContentEnd(entries, lastRowIndex) + 1
        }
        else -> tableRows[position].second
    }
}

/**
 * Finds all row entries belonging to a specific table using a single efficient pass.
 */
internal fun findTableRows(
    entries: List<LazyEntry>, 
    tableEntry: TableEntry, 
    tableIndex: Int
): List<Pair<RowEntry, Int>> {
    val tableRows = mutableListOf<Pair<RowEntry, Int>>()
    
    for (i in (tableIndex + 1) until entries.size) {
        when (val entry = entries[i]) {
            is RowEntry -> {
                if (entry.table.id == tableEntry.table.id && entry.level == tableEntry.level) {
                    tableRows.add(entry to i)
                } else if (entry.level <= tableEntry.level && entry.table.id != tableEntry.table.id) {
                    break // Reached a different table at same/lower level
                }
            }
            is TableEntry -> {
                if (entry.level <= tableEntry.level && entry.table.id != tableEntry.table.id) {
                    break // Reached a different table at same/lower level
                }
            }
        }
    }
    
    return tableRows
}

/**
 * Finds the end index of all content belonging to a specific row (including nested content).
 */
internal fun findRowContentEnd(entries: List<LazyEntry>, rowIndex: Int): Int {
    val rowEntry = entries[rowIndex] as? RowEntry ?: return rowIndex
    var endIndex = rowIndex

    for (i in (rowIndex + 1) until entries.size) {
        val entry = entries[i]
        if (entry.level <= rowEntry.level) break
        endIndex = i
    }

    return endIndex
}

/**
 * Updates row numbers for all rows belonging to a specific table.
 */
internal fun updateRowNumbers(entries: MutableList<LazyEntry>, tableId: String) {
    var rowNumber = 0
    
    for (i in entries.indices) {
        val entry = entries[i]
        if (entry is RowEntry && entry.table.id == tableId) {
            entries[i] = entry.copy(number = ++rowNumber)
        }
    }
}