package joyfill.editors.chart.internal

import joyfill.IdentityGenerator
import joyfill.chart.Line
import joyfill.chart.Point
import joyfill.editors.chart.LineEditor
import joyfill.editors.chart.PointEditor

@PublishedApi
internal class LineEditorImpl(
    private val identity: IdentityGenerator,
    private val lineId: String,
    private val getLine: () -> Line?,
    private val onValueChange: (() -> Unit)? = null,
) : LineEditor {

    private val emptyLine = Line(id = identity.generate(), title = "")

    override val line: Line
        get() = getLine() ?: emptyLine

    override fun title(value: String) {
        line.title = value
        onValueChange?.invoke()
    }

    override fun title() = line.title.orEmpty()

    override fun description(value: String?) {
        line.description = value
        onValueChange?.invoke()
    }

    override fun description() = line.description

    override fun points(): List<PointEditor> = line.points.map {
        PointEditorImpl(it, onValueChange)
    }

    private fun look(key: String): Point? = line.points.find { it.id == key || it.label == key }

    override fun find(key: String): PointEditor? {
        val point = look(key) ?: return null
        return PointEditorImpl(point, onValueChange)
    }

    override fun point(p: Point): PointEditor {
        return PointEditorImpl(p, onValueChange)
    }

    override fun add(): PointEditor {
        val currentLine = line // Get current line from collection
        val point = Point(
            id = identity.generate(),
            label = "Point ${currentLine.points.size + 1}",
        )

        currentLine.points.add(point)
        onValueChange?.invoke()
        return PointEditorImpl(point, onValueChange)
    }

    override fun remove(key: String): Point? {
        val currentLine = line // Get current line from collection
        val point = look(key) ?: return null
        currentLine.points.remove(point)
        onValueChange?.invoke()
        return point
    }
}