package joyfill.collections.internal

import joyfill.Document
import joyfill.Field
import joyfill.IdentityGenerator
import joyfill.Page
import joyfill.collections.FieldCollection
import joyfill.collections.PageCollection
import joyfill.editors.chart.ChartEditor
import joyfill.editors.collection.CollectionEditor
import joyfill.editors.components.ComponentEditor
import joyfill.editors.components.internal.AbstractComponentEditorFinder
import joyfill.editors.components.internal.toEditor
import joyfill.editors.table.HiddenReason
import joyfill.editors.table.TableEditor
import joyfill.events.ChangeEvent
import joyfill.events.EventDispatcher
import wisdom.ResolutionResource
import wisdom.ast.Library

internal class FieldCollectionImpl(
    private val document: Document,
    private val pages: PageCollection,
    private val identity: IdentityGenerator,
    private val resolver: ResolutionResource,
    private val library: Library?,
    private val onChange: ((ChangeEvent) -> Unit)?,
    private val circularFields: Set<String> = emptySet(),
) : AbstractComponentEditorFinder(), FieldCollection, EventDispatcher {

    override fun all() = document.fields.map { it.toEditor() }

    override fun from(page: String): List<ComponentEditor> {
        val files = document.files
        val pages = files.flatMap { it.views }.flatMap { it.pages } + files.flatMap { it.pages }
        val p = pages.find { it.identifier == page || it.id == page || it.name == page }
            ?: return emptyList()
        return from(p)
    }

    private fun look(key: String): Field? = document.fields.find { field ->
        key == field.identifier || key == field.id || key.equals(field.title, ignoreCase = true)
    }

    override fun from(page: Page): List<ComponentEditor> {
        val positionMap = page.positions.associateBy { it.field }
        val positionIds = positionMap.keys

        return document.fields
            .filter {
                it.id in positionIds
            }.sortedWith(
                compareBy(
                    { df -> positionMap[df.id]?.y },
                    { df -> positionMap[df.id]?.x }
                )
            ).map { it.toEditor() }
    }

    internal fun getNonHiddenFieldFrom(page: Page): List<ComponentEditor> {
        val positionMap = page.positions.associateBy { it.field }
        val positionIds = positionMap.keys

        return document.fields
            .filter {
                it.id in positionIds && !it.hidden
            }
            .sortedWith(
                compareBy(
                    { df -> positionMap[df.id]?.y },
                    { df -> positionMap[df.id]?.x }
                )
            )
            .map { it.toEditor() }
    }

    override fun find(key: String?): ComponentEditor? {
        val field = look(key ?: return null)
        return field?.toEditor()
    }

    private fun Field.dependents(): List<ComponentEditor> {
        val fields = document.fields
        return buildList {
            val descMap = document.formulas.associateBy({ it.desc }, { it })
            val idMap = document.formulas.associateBy({ it.id }, { it })
            val formulaLookup = descMap + idMap

            for (field in fields) {
                // Skip self-reference to prevent circular dependency
                if (field.id == this@dependents.id) continue

                val formulas = field.formulas.mapNotNull { application ->
                    formulaLookup[application.formula]
                }

                if (formulas.any { it.expression.contains(id) }) {
                    // Use cache to avoid circular dependency
                    val editor = cache[field.id] ?: field.toEditorForDependency()
                    add(editor)
                }
            }
        }
    }

    private val cache = mutableMapOf<String, ComponentEditor>()

    init {
        //TODO make each field resolve its condition on its own
        all().forEach { it.resolveConditions() }
    }

    private fun Field.toEditor(): ComponentEditor = toEditor(
        cache = cache,
        document = document,
        identity = identity,
        resolver = resolver,
        pages = pages,
        library = library,
        onChange = onChange,
        dispatcher = this@FieldCollectionImpl,
        dependents = { dependents() },
        circularFields = circularFields
    )

    // Special version for dependency resolution that provides empty dependents to avoid infinite recursion
    private fun Field.toEditorForDependency(): ComponentEditor = toEditor(
        cache = cache,
        document = document,
        identity = identity,
        resolver = resolver,
        pages = pages,
        library = library,
        onChange = onChange,
        dispatcher = this@FieldCollectionImpl,
        dependents = { emptyList() }, // Empty dependents to break circular dependency
        circularFields = circularFields
    ).also { cache[id] = it }

    override fun table(key: String?): TableEditor? = find(key) as? TableEditor

    override fun collection(key: String?): CollectionEditor? = find(key) as? CollectionEditor

    override fun chart(key: String?): ChartEditor? = find(key) as? ChartEditor

    override fun hide(component: String) {
        find(component)?.hide()
    }

    override fun show(component: String) {
        find(component)?.show()
    }

    override fun setHidden(component: String, reason: HiddenReason, value: Boolean) {
        find(component)?.setHiddenTo(value, reason, true)
    }
}