package joyfill2.tools.validation

import joyfill2.table.Column
import joyfill2.table.Row
import joyfill2.table.TableComponent

sealed interface TableValidity : ComponentValidity {
    val rows: List<RowValidity>

    fun isValid(column: Column): Boolean
    fun isValid(row: Row): Boolean
}

data class TableValid(
    override val component: TableComponent,
    override val rows: List<RowValidity>
) : TableValidity, Valid {
    override val messages: List<String> = rows.flatMap { it.cells.values }.flatMap { it.messages }
    override fun isValid(row: Row): Boolean = true
    override fun isValid(column: Column) = true
}

data class TableInvalid(
    override val component: TableComponent,
    override val rows: List<RowValidity>,
    val valid: List<RowValidity>,
    val invalid: List<RowValidity>
) : TableValidity, Invalid {
    override val messages: List<String> = rows.flatMap { it.cells.values }.flatMap { it.messages }
    override fun isValid(row: Row): Boolean {
        return invalid.find { it.row.id == row.id } !is RowInvalid
    }

    private val columnCellCache: MutableMap<String, List<ComponentValidity>> = mutableMapOf()

    override fun isValid(column: Column): Boolean {
        val columnId = column.id
        val cells = columnCellCache.getOrPut(columnId) {
            invalid.flatMap { row -> row.cells.filterKeys { it.id == columnId }.values }
        }
        for (cell in cells) if (cell is ComponentInvalid) return false
        return true
    }
}