package joyfill2.editors.table.internal

import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.collections.PageCollection
import joyfill2.editors.table.Columns
import joyfill2.editors.table.RowManager
import joyfill2.editors.table.RowResult
import joyfill2.editors.table.TableEditor
import joyfill2.events.ChangeEvent
import joyfill2.events.Target
import joyfill2.table.Cell
import joyfill2.table.Row
import joyfill2.table.TableComponent
import joyfill2.tools.validation.TableInvalid
import joyfill2.tools.validation.TableValid
import joyfill2.tools.validation.TableValidity

internal class TableEditorImpl(
    override val component: TableComponent,
    identity: IdentityGenerator,
    document: Document,
    pages: PageCollection,
    onChange: ((ChangeEvent) -> Unit)?,
    identifier: String,
) : AbstractTableEditor(component, document, pages,onChange, identifier, mapOf(), component.id), TableEditor {

    override val columns: Columns by lazy { ColumnsImpl(component) }

    override fun getCell(column: String): Cell? = null

    override val rows by lazy {
        RowsImpl(
            component = component,
            document = document,
            identity = identity,
            onAppend = ::onAppendRow,
            onMove = ::onMove,
            onDelete = ::onDelete,
            onAddAfter = ::onAddAfter,
            next = { next() },
            prev = { prev() },
            onChange = ::onRowChange,
            identifier = identifier,
        )
    }

    override fun resolveConditions() {}

    override fun validate(): TableValidity {
        val validity = when {
            component.required && component.value.isEmpty() ->
                TableInvalid(component, emptyList(), emptyList(), emptyList())
            else ->
                TableValid(component, emptyList())
        }
        return validity
    }

    init {
        state.value = state.value.copy(
            validity = validate()
        )
    }

    private fun onRowChange(changeEvent: ChangeEvent, row: Row) {
        changeEvent.changelogs.forEach { changeLog ->
            val value = buildRowUpdateChangeValue(row, changeLog)
            notifyChange(
                value = value,
                target = Target.field_value_rowUpdate
            )
        }
        resolveConditions()
    }

    private fun onAppendRow(manager: RowManager) {
        if (manager.row.id in state.value.rows.map { it.row.id }) return
        val value = buildRowCreateChangeValue(manager, component.value.lastIndex)
        notifyChange(
            value = value,
            target = Target.field_value_rowCreate
        )
        state.value = state.value.copy(
            rows = state.value.rows + manager,
            validity = validate()
        )
    }

    private fun onMove(rowResult: RowResult<Row>) {
        if (rowResult.index < 0) return
        val value = buildRowMoveChangeValue(rowResult.row, rowResult.index)
        notifyChange(
            value = value,
            target = Target.field_value_rowMove
        )

        val newRows = component.value.mapNotNull { row ->
            state.value.rows.find { it.row.id == row.id }
        }

        state.value = state.value.copy(rows = newRows)
    }

    private fun onDelete(keys: List<String>) {
        val removed = state.value.rows.filter { it.row.id in keys }
        val retained = state.value.rows - removed.toSet()

        removed.forEach {
            val value = buildRowDeleteChangeValue(it)
            notifyChange(
                value = value,
                target = Target.field_value_rowDelete
            )
        }

        state.value = state.value.copy(
            rows = retained,
            validity = validate()
        )
        unselectAll()
    }

    private fun onAddAfter(id: String, manager: RowManager?) {
        if (manager == null) return
        val index = state.value.rows.indexOfFirst { it.row.id == id }
        if (index != -1) {
            val updatedRows = state.value.rows.toMutableList()
            updatedRows.add(index + 1, manager)

            val value = buildRowCreateChangeValue(manager, index + 1)
            notifyChange(
                value = value,
                target = Target.field_value_rowCreate
            )

            state.value = state.value.copy(rows = updatedRows)
            unselectAll()
        }
    }
}