package joyfill2.editors.table.internal

import joyfill2.Document
import joyfill2.IdentityGenerator
import joyfill2.editors.table.RowManager
import joyfill2.editors.table.RowResult
import joyfill2.editors.table.intiCells
import joyfill2.events.ChangeEvent
import joyfill2.table.Row
import joyfill2.table.TableComponent
import joyfill2.table.toTableRow
import joyfill2.utils.ID

internal class RowsImpl(
    private val component: TableComponent,
    private val document: Document,
    private val identifier: String,
    private val identity: IdentityGenerator,
    private val onAppend: (RowManager) -> Unit,
    private val onDelete: (List<String>) -> Unit,
    private val onAddAfter: (String, RowManager?) -> Unit,
    private val onMove: (RowResult<Row>) -> Unit,
    private val onChange: ((ChangeEvent, Row) -> Unit)?,
    private val next: () -> RowManager?,
    private val prev: () -> RowManager?,
) : AbstractRows<RowManager>(component, component.rowOrder) {

    override fun create(empty: Boolean): RowManager {
        val row = mutableMapOf<String, Any?>(
            ID to identity.generate(),
            Row::deleted.name to false,
            Row::cells.name to component.intiCells(empty).toMap()
        ).toTableRow(component.columns)

        val manager = RowManagerImpl(
            row = row,
            columns = component.columns,
            document = document,
            identity = identity,
            onChange = {
                onChange?.invoke(it, row)
            },
            next = next,
            prev = prev,
            fieldId = component.id,
            identifier = identifier
        )
        return manager
    }

    override fun append(): RowManager {
        val manager = super.append()
        onAppend(manager)
        return manager
    }

    override fun delete(keys: List<String>): List<RowManager> {
        val result = super.delete(keys)
        onDelete(keys)
        return result
    }

    override fun addAfter(id: String): RowManager? {
        val manager = super.addAfter(id)
        onAddAfter(id, manager)
        return manager
    }

    override fun down(id: String, by: Int): RowResult<Row> {
        val result = super.down(id, by)
        onMove(result)
        return result
    }

    override fun up(id: String, by: Int): RowResult<Row> {
        val result = super.up(id, by)
        onMove(result)
        return result
    }

    override fun find(id: String): RowManager? = cache.getOrPut(id) {
        val row = component.value.find { it.id == id } ?: return null
        row.toManager()
    }

    override fun Row.toManager() = RowManagerImpl(
        row = this,
        columns = component.columns,
        document = document,
        identity = identity,
        onChange = {
            onChange?.invoke(it, this)
        },
        next = next,
        prev = prev,
        fieldId = component.id,
        identifier = identifier
    )
}